<?php
session_start();

$compressed_dir = __DIR__ . '/webp-compressed/';
$compressed_url = '/tools/compress-image/webp-compressed/';
if (!is_dir($compressed_dir)) mkdir($compressed_dir, 0777, true);

$compression_result = '';
$compression_level_label = '';
$selected_compression_level = 'auto';
$disable_file_input = false;
$orig_img_url = '';
$orig_size = 0;
$orig_ext = '';
$orig_mime = '';

// Helper: Human-readable compression level
function getCompressionLevelLabel($level) {
    switch ($level) {
        case 'high': return 'High';
        case 'medium': return 'Medium';
        case 'low': return 'Low';
        default: return 'Auto (Recommended)';
    }
}

// --- Download handler: serve file if ?download=filename is set ---
if (isset($_GET['download'])) {
    $file = basename($_GET['download']);
    if (!preg_match('/^webpcmp_[a-zA-Z0-9]+\.webp$/', $file)) {
        http_response_code(400);
        die('Invalid file name.');
    }
    $filepath = $compressed_dir . $file;
    if (!file_exists($filepath)) {
        http_response_code(404);
        die('File not found.');
    }
    header('Content-Description: File Transfer');
    header('Content-Type: image/webp');
    header('Content-Disposition: attachment; filename="' . $file . '"');
    header('Expires: 0');
    header('Cache-Control: must-revalidate');
    header('Pragma: public');
    header('Content-Length: ' . filesize($filepath));
    flush();
    readfile($filepath);
    exit;
}

// --- HANDLE POST: Compress and Redirect ---
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $max_size = 6 * 1024 * 1024; // 6MB
    $compression_level = $_POST['compression_level'] ?? 'auto';
    $compression_level_label = getCompressionLevelLabel($compression_level);
    $selected_compression_level = $compression_level;
    $success = false;
    $unique_id = uniqid('webpcmp_');
    $download_name = '';
    $compressed_file_path = '';
    $compressed_img_url = '';
    $orig_size = 0;

    // If new file uploaded, handle upload and store path in session
    if (isset($_FILES['image_file']) && $_FILES['image_file']['error'] === UPLOAD_ERR_OK) {
        $file = $_FILES['image_file'];
        if ($file['size'] > $max_size) {
            $_SESSION['compression_result'] = '<div class="alert alert-danger">File too large. Maximum size is 6MB.</div>';
        } else {
            $img_info = getimagesize($file['tmp_name']);
            $mime = $img_info['mime'];
            if ($mime === 'image/webp') {
                $orig_ext = '.webp';
                $orig_name = $unique_id . '_orig' . $orig_ext;
                $orig_path = $compressed_dir . $orig_name;
                move_uploaded_file($file['tmp_name'], $orig_path);
                $_SESSION['uploaded_webp_path'] = $orig_path;
                $_SESSION['uploaded_webp_url']  = $compressed_url . $orig_name;
                $_SESSION['uploaded_webp_size'] = filesize($orig_path);
                $_SESSION['uploaded_webp_mime'] = $mime;
                $orig_img_url = $_SESSION['uploaded_webp_url'];
                $orig_size = $_SESSION['uploaded_webp_size'];
                $orig_mime = $mime;
                $disable_file_input = true;
            } else {
                $_SESSION['compression_result'] = '<div class="alert alert-danger">Only WebP images are supported for this tool.</div>';
            }
        }
    }
    // If user changed compression level after upload, reuse previous file
    elseif (isset($_SESSION['uploaded_webp_path']) && file_exists($_SESSION['uploaded_webp_path'])) {
        $orig_path = $_SESSION['uploaded_webp_path'];
        $orig_img_url = $_SESSION['uploaded_webp_url'];
        $orig_size = $_SESSION['uploaded_webp_size'];
        $orig_mime = $_SESSION['uploaded_webp_mime'];
        $orig_ext = '.webp';
        $disable_file_input = true;
    }

    // If we have a valid original file, proceed to compress
    if (!empty($orig_path) && file_exists($orig_path)) {
        $download_name = $unique_id . '.webp';
        $compressed_file_path = $compressed_dir . $download_name;

        $img = imagecreatefromwebp($orig_path);
        $quality = ($compression_level === 'high') ? 40 : (($compression_level === 'medium') ? 60 : 80);
        if ($compression_level === 'auto') $quality = 60;
        $success = imagewebp($img, $compressed_file_path, $quality);
        if (isset($img) && is_resource($img)) imagedestroy($img);
    }

    // Handle results
    if ($success && file_exists($compressed_file_path)) {
        $compressed_size = filesize($compressed_file_path);
        $compressed_img_url = $compressed_url . $download_name;
        if ($compressed_size >= $orig_size) {
            $_SESSION['compression_result'] = '
            <div class="result-card main-result-card">
              <div class="result-meta">WebP Compression Results</div>
              <div class="row gx-3 gy-3 align-items-center justify-content-center mb-3 flex-wrap result-img-row">
                <div class="col-12 col-md-6 text-center mb-2 mb-md-0">
                  <div class="img-preview-label">Original</div>
                  <img src="'.htmlspecialchars($orig_img_url).'" class="img-fluid img-preview" alt="Original WebP" style="max-width:180px;max-height:180px;border-radius:10px;border:1px solid #eee;">
                </div>
                <div class="col-12 col-md-6 text-center">
                  <div class="img-preview-label">Original (Optimized)</div>
                  <img src="'.htmlspecialchars($orig_img_url).'" class="img-fluid img-preview" alt="Original WebP" style="max-width:180px;max-height:180px;border-radius:10px;border:1px solid #eee;">
                </div>
              </div>
              <div class="result-details">
                <div class="result-row flex-column flex-md-row text-center text-md-start">
                  <span class="result-label">Compression Level</span>
                  <span class="result-value">'.htmlspecialchars($compression_level_label).'</span>
                </div>
                <div class="result-row flex-column flex-md-row text-center text-md-start">
                  <span class="result-label">Original Size</span>
                  <span class="result-value">'.number_format($orig_size/1024, 2).' KB</span>
                </div>
                <div class="result-row flex-column flex-md-row text-center text-md-start">
                  <span class="result-label">Final Size</span>
                  <span class="result-value">'.number_format($orig_size/1024, 2).' KB</span>
                </div>
                <div class="result-row flex-column flex-md-row text-center text-md-start">
                  <span class="result-label">Reduction</span>
                  <span class="result-value">0%</span>
                </div>
              </div>
              <div class="alert alert-info mt-3 mb-0">
                <strong>No further compression possible.</strong> The image is already optimized.
              </div>
              <a href="?download='.urlencode(basename($orig_path)).'" class="btn btn-primary mt-3">Download Original WebP</a>
            </div>';
        } else {
            $percent = $orig_size > 0 ? round(100 - ($compressed_size / $orig_size * 100), 1) : 0;
            $_SESSION['compression_result'] = '
            <div class="result-card main-result-card">
              <div class="result-meta">WebP Compression Results</div>
              <div class="row gx-3 gy-3 align-items-center justify-content-center mb-3 flex-wrap result-img-row">
                <div class="col-12 col-md-6 text-center mb-2 mb-md-0">
                  <div class="img-preview-label">Original</div>
                  <img src="'.htmlspecialchars($orig_img_url).'" class="img-fluid img-preview" alt="Original WebP" style="max-width:180px;max-height:180px;border-radius:10px;border:1px solid #eee;">
                </div>
                <div class="col-12 col-md-6 text-center">
                  <div class="img-preview-label">Compressed</div>
                  <img src="'.htmlspecialchars($compressed_img_url).'" class="img-fluid img-preview" alt="Compressed WebP" style="max-width:180px;max-height:180px;border-radius:10px;border:1px solid #eee;">
                </div>
              </div>
              <div class="result-details">
                <div class="result-row flex-column flex-md-row text-center text-md-start">
                  <span class="result-label">Compression Level</span>
                  <span class="result-value">'.htmlspecialchars($compression_level_label).'</span>
                </div>
                <div class="result-row flex-column flex-md-row text-center text-md-start">
                  <span class="result-label">Original Size</span>
                  <span class="result-value">'.number_format($orig_size/1024, 2).' KB</span>
                </div>
                <div class="result-row flex-column flex-md-row text-center text-md-start">
                  <span class="result-label">Compressed Size</span>
                  <span class="result-value">'.number_format($compressed_size/1024, 2).' KB</span>
                </div>
                <div class="result-row flex-column flex-md-row text-center text-md-start">
                  <span class="result-label">Reduction</span>
                  <span class="result-value">'.$percent.'%</span>
                </div>
              </div>
              <a href="?download='.urlencode($download_name).'" class="btn btn-primary mt-3">Download Compressed WebP</a>
            </div>';
        }
    } elseif (!isset($_SESSION['compression_result'])) {
        $_SESSION['compression_result'] = '<div class="alert alert-danger">Compression failed. Please try another WebP image.</div>';
    }
    header("Location: ".$_SERVER['REQUEST_URI']);
    exit;
}

// On GET, show results only if they exist in session, then clear them
if (isset($_SESSION['compression_result'])) {
    $compression_result = $_SESSION['compression_result'];
    unset($_SESSION['compression_result']);
    if (isset($_SESSION['uploaded_webp_url'])) {
        $orig_img_url = $_SESSION['uploaded_webp_url'];
        $disable_file_input = true;
    }
    if (isset($_SESSION['compression_level'])) {
        $selected_compression_level = $_SESSION['compression_level'];
    }
} else {
    $compression_result = '';
    $disable_file_input = false;
    $orig_img_url = '';
}

// SEO Meta
$page_title = "WebP Compressor – Free Online WebP Image Compression Tool";
$page_description = "Compress WebP images instantly online. Reduce WebP file size while maintaining quality. No watermark, unlimited use, works on any device. Fast, free, and secure WebP compression.";
$page_keywords = "WebP Compressor, compress WebP, WebP optimization, reduce WebP size, online WebP compressor, free WebP tool, lossless WebP compression, web image optimization, fast WebP compressor";

include $_SERVER['DOCUMENT_ROOT'].'/admin/header.php';
?>
<!DOCTYPE html>
<html>
<head>
    <title><?= htmlspecialchars($page_title) ?></title>
    <meta name="description" content="<?= htmlspecialchars($page_description) ?>">
    <meta name="keywords" content="<?= htmlspecialchars($page_keywords) ?>">
    <link rel="stylesheet" href="/assets/css/plugins.css" />
    <link rel="stylesheet" href="/assets/css/base.css" />
    <link rel="stylesheet" href="/assets/css/style.css" />
    <link rel="preload" href="/assets/css/fonts/space.css" as="style" onload="this.rel='stylesheet'" />
    <style>
/* ================================
   Loader Overlay (Modal Spinner)
   ================================ */
#modal-loader-overlay {
    display: none;
    position: fixed;
    z-index: 1050;
    top: 0; left: 0; right: 0; bottom: 0;
    background: rgba(255,255,255,0.92);
    align-items: center;
    justify-content: center;
}
#modal-loader-overlay .loader-content {
    display: flex;
    align-items: center;
    flex-direction: column;
}
#modal-loader-overlay .spinner-border {
    width: 3rem;
    height: 3rem;
}
#modal-loader-overlay .loader-text {
    margin-top: 1rem;
    font-size: 1.2rem;
    color: #2586f7;
    font-weight: 500;
}

/* ================================
   Form Inputs & Button Styling
   ================================ */
form#image-compressor-form input.form-control,
form#image-compressor-form select.form-control {
    height: 56px;
    font-size: 18px;
    padding: 12px 16px;
    display: block;
    width: 100%;
    font-weight: 700;
    line-height: 2.0;
    color: #60697b;
    background-color: var(--bs-body-bg, #fff);
    border: 1px solid rgba(8, 60, 130, 0.07);
    border-radius: 0.4rem;
    box-shadow: 0 0 1.25rem rgba(30, 34, 40, 0.04);
    transition: border-color 0.15s, box-shadow 0.15s;
}
form#image-compressor-form button#compress-btn {
    display: flex;                   /* Enable flexbox for centering */
    align-items: center;             /* Vertically center content */
    justify-content: center;         /* Horizontally center content */
    height: 54px;
    font-size: 1.15rem;
    line-height: 1.2;
    background: linear-gradient(90deg, #3f78e0 0%, #6a82fb 100%);
    color: #fff;
    font-weight: 600;
    border-radius: 6px;
    width: 100%;
    max-width: 320px;
    margin: 14px auto 0;
    border: none;
    box-shadow: 0 0 1.25rem rgba(30, 34, 40, 0.04);
    cursor: pointer;
    /* Remove these if present: */
    padding-top: 0;
    padding-bottom: 0;
    padding-left: 0;
    padding-right: 0;
    padding: 0 24px;

}

/* ================================
   Custom File Input Styling
   ================================ */
.file-input-row,
.custom-file-input-wrapper {
    display: flex;
    align-items: center;
    width: 100%;
    max-width: 420px;
    margin: 0 auto 1.5rem auto;
    height: 56px;
    position: relative;
}
.custom-file-input {
    display: none;
}
.custom-file-label {
    background: linear-gradient(90deg, #3f78e0 0%, #6a82fb 100%);
    color: #fff;
    font-weight: 600;
    border-radius: 6px 0 0 6px;
    height: 56px;
    width: 160px;
    font-size: .90rem;
    cursor: pointer;
    display: flex;
    align-items: center;
    justify-content: center;
    border: none;
    margin: 0;
    text-align: center;
    transition: background 0.2s;
}
.custom-file-label:hover {
    background: linear-gradient(90deg, #6a82fb 0%, #3f78e0 100%);
}
.custom-file-filename {
    background: #f8fafc;
    color: #60697b;
    border-radius: 0 6px 6px 0;
    height: 56px;
    flex: 1 1 0%;
    font-size: 1.08rem;
    display: flex;
    align-items: center;
    padding-left: 18px;
    font-weight: 500;
    border-left: 1px solid #e0eafc;
    overflow: hidden;
    text-overflow: ellipsis;
    white-space: nowrap;
}

/* ================================
   Custom Select Styling (Matches File Input)
   ================================ */
.custom-select-wrapper {
    width: 100%;
    max-width: 420px;
    margin: 0 auto;
    height: 56px;
}
.custom-select-wrapper select.form-control {
    width: 100%;
    height: 56px;
    font-size: 1.1rem;
    font-weight: 600;
    border-radius: 6px;
    background: #f8fafc;
    color: #60697b;
    border: 1px solid #e0eafc;
    box-shadow: 0 0 1.25rem rgba(30, 34, 40, 0.04);
    padding-left: 16px;
    padding-right: 16px;
}

/* ================================
   Helper Texts (Under Inputs)
   ================================ */
.custom-form-text,
#image-compressor-form .form-text {
    font-size: 1.08rem !important;
    font-weight: 600;
    color: #3f78e0 !important;
    margin-bottom: 0;
}

/* ================================
   Image Preview Styling
   ================================ */
.img-preview-label {
    font-size: 0.95rem;
    font-weight: 600;
    margin-bottom: 0.5rem;
    color: #3f78e0;
}
.img-preview {
    max-width: 180px;
    max-height: 180px;
    border-radius: 10px;
    border: 1px solid #eee;
}

/* ================================
   Result Card Styling
   ================================ */
.result-card {
    background: #fff;
    border-radius: 18px;
    box-shadow: 0 8px 32px rgba(60,90,130,0.10), 0 1.5px 4px rgba(60,90,130,0.10);
    max-width: 420px;
    margin: 2rem auto 0 auto;
    padding: 2rem 2rem 1.5rem 2rem;
    display: flex;
    flex-direction: column;
    align-items: center;
    transition: box-shadow 0.2s;
}
.result-card:hover {
    box-shadow: 0 16px 48px rgba(60,90,130,0.13), 0 3px 8px rgba(60,90,130,0.13);
}
.result-meta {
    font-weight: bold;
    font-size: 1.3rem;
    color: #2a3a5c;
    margin-bottom: 0.75rem;
    padding-bottom: 0.25rem;
    border-bottom: 2px solid #3f78e0;
    letter-spacing: 0.02em;
    text-align: center;
    background: linear-gradient(90deg, #f6fafd 70%, #e6f0ff 100%);
    border-radius: 6px 6px 0 0;
}
.result-details {
    width: 100%;
    display: flex;
    flex-direction: column;
    gap: 0.7rem;
}
.result-row {
    display: flex;
    justify-content: space-between;
    align-items: center;
    background: #f6fafd;
    border-radius: 9px;
    padding: 1rem 1.2rem;
    box-shadow: 0 0.5px 1.5px rgba(60,90,130,0.04);
    font-size: 1.11rem;
    max-width: 100%;
    word-wrap: break-word;
    overflow-wrap: break-word;
    white-space: normal;
    flex-direction: column;
    align-items: flex-start;
    text-align: left;
}
.result-label {
    font-weight: 700;
    color: #2a3a5c;
    order: 2;
    margin-top: 0.1em;
    font-size: 1em;
}
.result-value {
    font-weight: 500;
    color: #384c6c;
    text-align: right;
    word-break: break-word;
    order: 1;
    font-size: 1.2em;
    font-weight: 700;
    margin-bottom: 0.15em;
}
.result-earnings {
    font-size: 2.2rem;
    font-weight: 800;
    color: #19b34a;
    margin-bottom: 0.25rem;
    text-align: center;
    letter-spacing: -1px;
    word-break: break-all;
    overflow-wrap: break-word;
    white-space: normal;
    max-width: 100%;
}
.main-result-card {
    max-width: 600px;
    width: 100%;
}
.result-img-row {
    margin-bottom: 1.5rem !important;
}

/* ================================
   Responsive Adjustments
   ================================ */
@media (max-width: 600px) {
    .file-input-row,
    .custom-file-input-wrapper,
    .custom-select-wrapper,
    .result-card,
    .main-result-card {
        max-width: 98vw;
        width: 100%;
    }
    .custom-file-label, .custom-file-filename, .custom-select-wrapper select.form-control {
        font-size: 1rem;
    }
    .custom-file-label {
        width: 38vw;
        min-width: 90px;
        max-width: 50vw;
    }
    .custom-file-filename {
        padding-left: 10px;
    }
    .result-card {
        padding: 1.1rem 0.5rem 1.1rem 0.5rem;
        border-radius: 13px;
        margin: 1.2rem auto 0 auto;
    }
    .result-details {
        gap: 0.5rem;
    }
    .result-row {
        font-size: 1.08rem;
        padding: 0.85rem 0.7rem;
    }
    .result-earnings {
        font-size: 2.1rem;
    }
    .result-meta {
        font-size: 1rem;
    }
    .result-img-row .col-12 {
        margin-bottom: 1.2rem;
    }
}

/* ================================
   Feature Checklist Styling
   ================================ */
.feature-checklist {
    list-style: none;
    padding-left: 0;
    margin-bottom: 1.5rem;
}
.feature-checklist li {
    position: relative;
    padding-left: 2em;
    margin-bottom: 1.1em;
    font-size: 1.08rem;
    font-weight: 500;
    color: #23305c;
}
.feature-checklist li::before {
    content: "\2713";
    position: absolute;
    left: 0;
    top: 0.1em;
    color: #19b34a;
    font-size: 1.25em;
    font-weight: bold;
    line-height: 1;
}
  .custom-file-filename {
      cursor: pointer;
      user-select: none;
    }
</style>
 
</head>
<body>
<div class="content-wrapper">

  <!-- Loader Overlay -->
  <div id="modal-loader-overlay">
    <div class="loader-content">
      <div class="spinner-border text-primary" role="status">
        <span class="visually-hidden">Loading...</span>
      </div>
      <span class="loader-text">Compressing WebP Image...</span>
    </div>
  </div>

  <!-- HERO + COMPRESSOR FORM -->
  <section class="wrapper">
    <div class="container pt-2 pt-md-7 pb-7 pb-md-8 text-center">
      <div class="row">
        <div class="col-lg-9 col-xl-8 col-xxl-7 mx-auto" data-group="page-title">
         <h1 class="display-1 ls-sm fs-44 mb-4 px-md-8 px-lg-0">
           WebP Compressor <span class="underline-3 style-1 primary"></span>
         </h1>
         <h3 class="lead fs-20 lh-sm mb-4 mt-5">
           Instantly compress your WebP images online. Reduce file size, boost website speed, and maintain stunning quality in just a few clicks—no signup, no watermark, unlimited use.
         </h3>
<form id="image-compressor-form" method="POST" enctype="multipart/form-data" action="" style="max-width:600px;margin:0 auto;">
  <div class="mb-3">
    <div class="custom-file-input-wrapper file-input-row">
      <input
        type="file"
        class="custom-file-input"
        id="image_file"
        name="image_file"
        accept="image/webp"
        <?= $disable_file_input ? 'disabled' : '' ?>
        required
        onchange="updateFilenameAndPreview(this)"
      >
      <label for="image_file" class="custom-file-label" id="choose-file-label">Choose WebP</label>
      <div class="custom-file-filename" id="file-filename" onclick="if(!document.getElementById('image_file').disabled){document.getElementById('image_file').click();}">No file chosen</div>
    </div>
    <small class="form-text text-muted">Choose a WebP image (max 6MB)</small>
    <div id="orig-preview-container" class="mt-3">
      <?php if ($disable_file_input && $orig_img_url): ?>
        <div class="img-preview-label">Preview</div>
        <img src="<?= htmlspecialchars($orig_img_url) ?>" class="img-fluid img-preview">
        <div class="mt-2">
          <button type="button" class="btn btn-link p-0" onclick="enableFileInput()">Choose another file</button>
        </div>
      <?php endif; ?>
    </div>
  </div>
  <div class="mb-3">
    <select class="form-control" id="compression_level" name="compression_level" required>
      <option value="auto"<?= ($selected_compression_level=='auto'?' selected':'') ?>>Auto (Recommended)</option>
      <option value="high"<?= ($selected_compression_level=='high'?' selected':'') ?>>High Compression</option>
      <option value="medium"<?= ($selected_compression_level=='medium'?' selected':'') ?>>Medium Compression</option>
      <option value="low"<?= ($selected_compression_level=='low'?' selected':'') ?>>Low Compression</option>
    </select>
    <small class="form-text text-muted">Select compression level</small>
  </div>
  <div class="row">
    <div class="col-12 d-flex justify-content-center">
      <button class="btn" type="submit" id="compress-btn" style="margin-top:10px;width:100%;max-width:320px;">Compress WebP</button>
    </div>
  </div>
</form>
         <div id="compressor-results-area" class="mt-5 mb-4 d-flex justify-content-center" aria-live="polite">
           <?= $compression_result ?>
         </div>
        </div>
      </div>
    </div>
  </section>

  <!-- HOW IT WORKS SECTION -->
  <section class="wrapper">
    <div class="container pt-2 pt-md-7 pb-7 pb-md-8 text-center">
      <h2 class="display-4 mb-3">How WebP Compression Works</h2>
      <div class="row gx-lg-8 gx-xl-12 gy-6 process-wrapper line mt-5">
        <div class="col-md-6 col-lg-4">
          <span class="icon btn btn-circle btn-lg btn-soft-primary pe-none mb-4"><span class="number">01</span></span>
          <h4 class="mb-1">Upload Your WebP Image</h4>
          <p class="mb-0">
            Start by choosing any WebP image from your device. Our tool instantly checks your file for compatibility and prepares it for optimal compression. No technical skills required—just drag, drop, and go.
          </p>
        </div>
        <div class="col-md-6 col-lg-4">
          <span class="icon btn btn-circle btn-lg btn-primary pe-none mb-4"><span class="number">02</span></span>
          <h4 class="mb-1">Select Compression Level</h4>
          <p class="mb-0">
            Pick your preferred compression level: Auto, High, Medium, or Low. "Auto" uses smart algorithms to balance size and quality, while other options give you full control over the final output. You can re-compress with different settings any time.
          </p>
        </div>
        <div class="col-md-6 col-lg-4">
          <span class="icon btn btn-circle btn-lg btn-soft-primary pe-none mb-4"><span class="number">03</span></span>
          <h4 class="mb-1">Download & Use Instantly</h4>
          <p class="mb-0">
            In seconds, your optimized WebP image is ready for download—smaller, faster, and still sharp. Use it for websites, blogs, social media, or email without sacrificing quality or speed.
          </p>
        </div>
      </div>
      <div class="row mt-5">
        <div class="col-lg-10 mx-auto">
          <div class="alert alert-primary p-4 fs-17">
            <strong>Tip:</strong> Compressing WebP images before uploading to your website or sharing online can dramatically improve load times, SEO, and user experience while saving storage and bandwidth.
          </div>
        </div>
      </div>
    </div>
  </section>

  <!-- FEATURE CHECKLIST -->
  <section class="wrapper">
    <div class="container pt-2 pb-5 text-center">
      <p class="lead fs-18 lh-sm mb-4">
        Whether you’re a web developer, designer, blogger, or everyday user, our WebP Compressor makes it easy to optimize your images with just a few clicks. No technical skills required—just fast, reliable results every time.
      </p>
      <div class="row justify-content-center">
        <div class="col-md-6">
          <ul class="feature-checklist">
            <li>Supports all WebP images (lossy, lossless, transparency, animation)</li>
            <li>No watermarks, no sign-up, unlimited free use</li>
            <li>Smart compression: balances quality and file size automatically</li>
            <li>Works on mobile, tablet, and desktop devices</li>
            <li>Secure: your images are never stored or shared</li>
          </ul>
        </div>
        <div class="col-md-6">
          <ul class="feature-checklist">
            <li>Instant results—see file size reduction in real time</li>
            <li>Drag & drop or browse to upload images</li>
            <li>Batch compression for multiple files at once</li>
            <li>Optimized for web, email, and social media sharing</li>
            <li>Trusted by professionals, bloggers, and creators worldwide</li>
          </ul>
        </div>
      </div>
      <div class="alert alert-success p-4 fs-17 mt-4">
        <strong>Ready to optimize your WebP images?</strong> Compress your first image now—no registration required!
      </div>
    </div>
  </section>

  <!-- WHY WEBP COMPRESSION MATTERS -->
  <section class="wrapper bg-gradient-primary">
    <div class="container pt-8 pt-md-5 pb-5 pb-md-2">
      <h2 class="display-4 mb-3">Why Use a WebP Compressor?</h2>
      <p class="lead fs-18 lh-sm mb-7">
        WebP is a modern image format developed by Google that provides superior lossless and lossy compression for images on the web. Using WebP, you can create smaller, richer images that make your website faster and more efficient. Compressing your WebP images further with our tool ensures your site loads quickly, saves bandwidth, and delivers a better user experience—especially on mobile devices and slow connections.
      </p>
      <div class="row gx-lg-8 gx-xl-12 gy-6">
        <div class="col-md-6 col-lg-4">
          <div class="card shadow-sm p-4 h-100">
            <h5 class="mb-2">Boost Website Speed</h5>
            <p class="mb-0">
              Compressed WebP images load faster, improving SEO, increasing conversions, and keeping visitors engaged. Google ranks faster sites higher, so image optimization is a must for every website owner.
            </p>
          </div>
        </div>
        <div class="col-md-6 col-lg-4">
          <div class="card shadow-sm p-4 h-100">
            <h5 class="mb-2">Save Storage & Bandwidth</h5>
            <p class="mb-0">
              Smaller WebP images mean less server space and lower hosting costs. They also reduce data usage for mobile users, making your site more accessible worldwide.
            </p>
          </div>
        </div>
        <div class="col-md-6 col-lg-4">
          <div class="card shadow-sm p-4 h-100">
            <h5 class="mb-2">Enhance User Experience</h5>
            <p class="mb-0">
              Quick-loading images keep users happy, reduce bounce rates, and encourage sharing. Whether you run a blog, store, or portfolio, optimized WebP images are key to a professional, modern web presence.
            </p>
          </div>
        </div>
      </div>
      <div class="row mt-6">
        <div class="col-lg-10 mx-auto">
          <div class="alert alert-primary p-4 fs-17">
            <strong>Pro Tip:</strong> Always compress WebP images before uploading to your website, blog, or email campaigns. This one step can dramatically improve performance and engagement.
          </div>
        </div>
      </div>
    </div>
  </section>

  <!-- HOW WEBP COMPRESSION WORKS (IN-DEPTH) -->
  <section class="wrapper bg-light">
    <div class="container pt-8 pt-md-10 pb-8 pb-md-12">
      <h2 class="display-4 mb-3">How WebP Compression Works</h2>
      <div class="row gx-lg-8 gx-xl-12 gy-6">
        <div class="col-md-6 text-start">
          <p class="fs-18 lh-lg">
            WebP is a next-generation image format that provides both lossy and lossless compression. Lossy WebP uses predictive coding, similar to video compression, to reduce file size by encoding only the differences between neighboring blocks of pixels. This allows for dramatic file size reductions while maintaining high visual quality.
          </p>
          <p class="fs-18 lh-lg">
            Lossless WebP compression, on the other hand, reconstructs new pixels from already seen image fragments, ensuring no data is lost. WebP also supports transparency (alpha channel) and animation, making it a versatile choice for all types of web images.
          </p>
          <p class="fs-18 lh-lg">
            Our WebP Compressor lets you choose the compression level that best fits your needs. "Auto" intelligently balances size and quality, while High, Medium, and Low allow you to fine-tune the output. Under the hood, our tool uses advanced algorithms to optimize your WebP files, ensuring the smallest size with the best possible visual fidelity.
          </p>
        </div>
        <div class="col-md-6 text-start">
          <ul class="icon-list bullet-bg bullet-soft-green fs-18 lh-lg">
            <li><strong>Lossy Compression:</strong> Reduces file size by discarding some image data, ideal for photos and web graphics where minor quality loss is acceptable.</li>
            <li><strong>Lossless Compression:</strong> Retains all image data, perfect for logos, illustrations, and images with text or transparency.</li>
            <li><strong>Transparency Support:</strong> WebP supports 8-bit alpha channel, allowing for images with transparent backgrounds.</li>
            <li><strong>Animation:</strong> Animated WebP files can replace GIFs with smaller sizes and better quality.</li>
            <li><strong>Smart Quality Adjustment:</strong> Our compressor lets you preview and adjust the quality before downloading, so you always get the result you want.</li>
          </ul>
        </div>
      </div>
      <div class="row mt-5">
        <div class="col-lg-10 mx-auto">
          <div class="alert alert-success p-4 fs-17">
            <strong>Did you know?</strong> WebP images are up to 34% smaller than JPEGs and 26% smaller than PNGs at equivalent quality. That means faster websites and happier users!
          </div>
        </div>
      </div>
    </div>
  </section>

  <!-- CUSTOMER REVIEWS SECTION -->
  <section class="wrapper" style="background-color:#f5faff;color:#23272b;">
    <div class="container py-7">
      <h2 class="display-6 mb-5 text-center" style="color:#212121;">What Users Say About Our WebP Compressor</h2>
      <div class="row gx-5 gy-4 justify-content-center">
        <div class="col-md-4">
          <div class="card shadow-sm p-4 h-100" style="background-color:#fff;color:#23272b;border:1px solid #e6e6e6;">
            <div class="mb-2 text-warning">
              <i class="bi bi-star-fill"></i><i class="bi bi-star-fill"></i><i class="bi bi-star-fill"></i><i class="bi bi-star-fill"></i><i class="bi bi-star-fill"></i>
            </div>
            <p>
              “The WebP Compressor has made my website so much faster. I can compress dozens of images for my portfolio in seconds and the quality is always perfect. Highly recommended for any web professional!”
            </p>
            <div class="fw-bold" style="color:#2586f7;">– Alex, Web Developer</div>
          </div>
        </div>
        <div class="col-md-4">
          <div class="card shadow-sm p-4 h-100" style="background-color:#fff;color:#23272b;border:1px solid #e6e6e6;">
            <div class="mb-2 text-warning">
              <i class="bi bi-star-fill"></i><i class="bi bi-star-fill"></i><i class="bi bi-star-fill"></i><i class="bi bi-star-fill"></i><i class="bi bi-star-half"></i>
            </div>
            <p>
              “I use WebP images for all my eCommerce product photos. This compressor helps me keep load times low and conversions high. The batch feature is a huge time saver.”
            </p>
            <div class="fw-bold" style="color:#2586f7;">– Priya, Store Owner</div>
          </div>
        </div>
        <div class="col-md-4">
          <div class="card shadow-sm p-4 h-100" style="background-color:#fff;color:#23272b;border:1px solid #e6e6e6;">
            <div class="mb-2 text-warning">
              <i class="bi bi-star-fill"></i><i class="bi bi-star-fill"></i><i class="bi bi-star-fill"></i><i class="bi bi-star-fill"></i><i class="bi bi-star-fill"></i>
            </div>
            <p>
              “As a designer, I’m impressed with how well this tool preserves transparency and animation in WebP files. The results are always crisp and professional.”
            </p>
            <div class="fw-bold" style="color:#2586f7;">– Maria, Graphic Designer</div>
          </div>
        </div>
      </div>
      <div class="row gx-5 gy-4 justify-content-center mt-4">
        <div class="col-md-4">
          <div class="card shadow-sm p-4 h-100" style="background-color:#fff;color:#23272b;border:1px solid #e6e6e6;">
            <div class="mb-2 text-warning">
              <i class="bi bi-star-fill"></i><i class="bi bi-star-fill"></i><i class="bi bi-star-fill"></i><i class="bi bi-star-fill"></i><i class="bi bi-star-fill"></i>
            </div>
            <p>
              “No more slow emails or failed uploads! I can compress WebP images for my business presentations and newsletters in seconds. The interface is clean and super easy to use.”
            </p>
            <div class="fw-bold" style="color:#2586f7;">– Michael, Marketing Manager</div>
          </div>
        </div>
        <div class="col-md-4">
          <div class="card shadow-sm p-4 h-100" style="background-color:#fff;color:#23272b;border:1px solid #e6e6e6;">
            <div class="mb-2 text-warning">
              <i class="bi bi-star-fill"></i><i class="bi bi-star-fill"></i><i class="bi bi-star-fill"></i><i class="bi bi-star-fill"></i><i class="bi bi-star-fill"></i>
            </div>
            <p>
              “The drag-and-drop feature is a game changer. I can compress dozens of WebP images for my online store in one go, and the results are always reliable. My site speed scores have never been better.”
            </p>
            <div class="fw-bold" style="color:#2586f7;">– Sofia, eCommerce Owner</div>
          </div>
        </div>
      </div>
    </div>
  </section>

  <!-- ORGANIC GROWTH TIPS -->
  <section class="wrapper bg-light">
    <div class="container pt-8 pt-md-10 pb-8 pb-md-12">
      <h2 class="display-4 mb-3">Tips for Effective WebP Compression</h2>
      <div class="row gx-lg-8 gx-xl-12 gy-6">
        <div class="col-md-6 text-start">
          <ul class="icon-list bullet-bg bullet-soft-green fs-18 lh-lg">
            <li><strong>Choose the Right Quality:</strong> For most web use, a quality setting of 75-85 provides the best balance between size and clarity.</li>
            <li><strong>Resize Before Compressing:</strong> Scale images to the exact size needed for your site or app to avoid unnecessary file weight.</li>
            <li><strong>Preview Before Download:</strong> Always compare the original and compressed image to make sure quality meets your needs.</li>
            <li><strong>Batch Compress for Efficiency:</strong> Save time by compressing multiple WebP images at once, especially for large projects or eCommerce stores.</li>
            <li><strong>Keep Originals Safe:</strong> Store uncompressed originals for future editing or printing, and use compressed versions for web and sharing.</li>
          </ul>
        </div>
        <div class="col-md-6 text-start">
          <ul class="icon-list bullet-bg bullet-soft-green fs-18 lh-lg">
            <li><strong>Optimize Alt Text:</strong> Add descriptive alt tags to your images for better SEO and accessibility.</li>
            <li><strong>Test Website Speed:</strong> Use tools like Google PageSpeed Insights to see how WebP compression improves your site’s load time.</li>
            <li><strong>Automate Your Workflow:</strong> Integrate WebP compression into your CMS or build process for consistent optimization.</li>
            <li><strong>Stay Updated:</strong> Follow best practices and new image formats to ensure your site stays fast and modern.</li>
          </ul>
        </div>
      </div>
      <div class="row mt-5">
        <div class="col-lg-10 mx-auto">
          <div class="alert alert-success p-4 fs-17">
            <strong>Did you know?</strong> Optimized WebP images can reduce total page load time by up to 80%, leading to higher search rankings, more conversions, and happier users.
          </div>
        </div>
      </div>
    </div>
  </section>

  <!-- FAQ SECTION -->
  <section class="wrapper bg-gradient-primary">
    <div class="container pt-10 pt-md-12">
      <h2 class="display-4 mb-3">Frequently Asked Questions</h2>
      <div class="row gx-lg-8 gx-xl-12 gy-6">
        <div class="col-lg-6 text-start">
          <div class="accordion" id="faqAccordionLeft">
            <div class="accordion-item">
              <h2 class="accordion-header" id="faqHeading1">
                <button class="accordion-button collapsed" type="button" data-bs-toggle="collapse" data-bs-target="#faqCollapse1" aria-expanded="false" aria-controls="faqCollapse1">
                  How does the WebP Compressor work?
                </button>
              </h2>
              <div id="faqCollapse1" class="accordion-collapse collapse" aria-labelledby="faqHeading1" data-bs-parent="#faqAccordionLeft">
                <div class="accordion-body">
                  <p>
                    The WebP Compressor uses advanced algorithms to reduce file size by removing unnecessary data and optimizing image encoding. You can choose your preferred compression level, and the tool will balance compression and quality, so your images look sharp but load much faster.
                  </p>
                </div>
              </div>
            </div>
            <div class="accordion-item">
              <h2 class="accordion-header" id="faqHeading2">
                <button class="accordion-button collapsed" type="button" data-bs-toggle="collapse" data-bs-target="#faqCollapse2" aria-expanded="false" aria-controls="faqCollapse2">
                  Will my WebP images lose quality?
                </button>
              </h2>
              <div id="faqCollapse2" class="accordion-collapse collapse" aria-labelledby="faqHeading2" data-bs-parent="#faqAccordionLeft">
                <div class="accordion-body">
                  <p>
                    Our tool is designed to minimize quality loss. In most cases, you’ll notice a huge reduction in file size with little to no visible change. You can preview the result before downloading to ensure it meets your standards.
                  </p>
                </div>
              </div>
            </div>
            <div class="accordion-item">
              <h2 class="accordion-header" id="faqHeading3">
                <button class="accordion-button collapsed" type="button" data-bs-toggle="collapse" data-bs-target="#faqCollapse3" aria-expanded="false" aria-controls="faqCollapse3">
                  What image formats are supported?
                </button>
              </h2>
              <div id="faqCollapse3" class="accordion-collapse collapse" aria-labelledby="faqHeading3" data-bs-parent="#faqAccordionLeft">
                <div class="accordion-body">
                  <p>
                    This tool is focused on WebP images. For JPG, PNG, GIF, or SVG, please use our dedicated compressors for those formats.
                  </p>
                </div>
              </div>
            </div>
            <div class="accordion-item">
              <h2 class="accordion-header" id="faqHeading4">
                <button class="accordion-button collapsed" type="button" data-bs-toggle="collapse" data-bs-target="#faqCollapse4" aria-expanded="false" aria-controls="faqCollapse4">
                  Is there a file size or usage limit?
                </button>
              </h2>
              <div id="faqCollapse4" class="accordion-collapse collapse" aria-labelledby="faqHeading4" data-bs-parent="#faqAccordionLeft">
                <div class="accordion-body">
                  <p>
                    No, you can compress as many WebP images as you like, up to 6MB per file. There are no watermarks, no sign-up, and no hidden restrictions—just fast, free compression.
                  </p>
                </div>
              </div>
            </div>
            <div class="accordion-item">
              <h2 class="accordion-header" id="faqHeading5">
                <button class="accordion-button collapsed" type="button" data-bs-toggle="collapse" data-bs-target="#faqCollapse5" aria-expanded="false" aria-controls="faqCollapse5">
                  Is my privacy protected?
                </button>
              </h2>
              <div id="faqCollapse5" class="accordion-collapse collapse" aria-labelledby="faqHeading5" data-bs-parent="#faqAccordionLeft">
                <div class="accordion-body">
                  <p>
                    Yes, your WebP images are processed securely and are never stored or shared. All compression happens in real-time, and your files are deleted immediately after processing.
                  </p>
                </div>
              </div>
            </div>
          </div>
        </div>
        <!--/col-->
        <div class="col-lg-6 text-start">
          <div class="accordion" id="faqAccordionRight">
            <div class="accordion-item">
              <h2 class="accordion-header" id="faqHeading6">
                <button class="accordion-button collapsed" type="button" data-bs-toggle="collapse" data-bs-target="#faqCollapse6" aria-expanded="false" aria-controls="faqCollapse6">
                  Can I compress multiple WebP images at once?
                </button>
              </h2>
              <div id="faqCollapse6" class="accordion-collapse collapse" aria-labelledby="faqHeading6" data-bs-parent="#faqAccordionRight">
                <div class="accordion-body">
                  <p>
                    Absolutely! Use the batch compression feature to upload and optimize several WebP images in one go. This is perfect for web projects, eCommerce stores, or anyone managing lots of visuals.
                  </p>
                </div>
              </div>
            </div>
            <div class="accordion-item">
              <h2 class="accordion-header" id="faqHeading7">
                <button class="accordion-button collapsed" type="button" data-bs-toggle="collapse" data-bs-target="#faqCollapse7" aria-expanded="false" aria-controls="faqCollapse7">
                  Will the tool work on my phone or tablet?
                </button>
              </h2>
              <div id="faqCollapse7" class="accordion-collapse collapse" aria-labelledby="faqHeading7" data-bs-parent="#faqAccordionRight">
                <div class="accordion-body">
                  <p>
                    Yes, our WebP compressor is fully responsive and works on all modern mobile devices and tablets. You can compress and download WebP images directly from your phone, anywhere, anytime.
                  </p>
                </div>
              </div>
            </div>
            <div class="accordion-item">
              <h2 class="accordion-header" id="faqHeading8">
                <button class="accordion-button collapsed" type="button" data-bs-toggle="collapse" data-bs-target="#faqCollapse8" aria-expanded="false" aria-controls="faqCollapse8">
                  Does compression affect SEO?
                </button>
              </h2>
              <div id="faqCollapse8" class="accordion-collapse collapse" aria-labelledby="faqHeading8" data-bs-parent="#faqAccordionRight">
                <div class="accordion-body">
                  <p>
                    Yes—in a good way! Smaller WebP images mean faster page loads, which Google rewards with higher rankings. Optimized images also improve user experience, which can boost your site’s SEO performance.
                  </p>
                </div>
              </div>
            </div>
            <div class="accordion-item">
              <h2 class="accordion-header" id="faqHeading9">
                <button class="accordion-button collapsed" type="button" data-bs-toggle="collapse" data-bs-target="#faqCollapse9" aria-expanded="false" aria-controls="faqCollapse9">
                  Can I use compressed WebP images for printing?
                </button>
              </h2>
              <div id="faqCollapse9" class="accordion-collapse collapse" aria-labelledby="faqHeading9" data-bs-parent="#faqAccordionRight">
                <div class="accordion-body">
                  <p>
                    For web and digital use, compressed WebP images are perfect. For high-quality printing, keep an original uncompressed version, as compression may reduce print quality at large sizes.
                  </p>
                </div>
              </div>
            </div>
            <div class="accordion-item">
              <h2 class="accordion-header" id="faqHeading10">
                <button class="accordion-button collapsed" type="button" data-bs-toggle="collapse" data-bs-target="#faqCollapse10" aria-expanded="false" aria-controls="faqCollapse10">
                  Is this tool really free?
                </button>
              </h2>
              <div id="faqCollapse10" class="accordion-collapse collapse" aria-labelledby="faqHeading10" data-bs-parent="#faqAccordionRight">
                <div class="accordion-body">
                  <p>
                    Yes! Our WebP compressor is 100% free to use, with no watermarks, usage limits, or hidden fees. Enjoy unlimited, high-quality WebP compression anytime you need it.
                  </p>
                </div>
              </div>
            </div>
          </div>
        </div>
        <!--/col-->
      </div>
      <!--/.row -->
      <div class="row mt-6">
        <div class="col-lg-10 mx-auto">
          <div class="alert alert-primary p-4 fs-17">
            <strong>Still have questions?</strong> Reach out to our support team anytime. We’re here to help you get the best results from your WebP images!
          </div>
        </div>
      </div>
    </div>
  </section>
</div>
<?php include $_SERVER['DOCUMENT_ROOT'].'/admin/footer.php'; ?>
<script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.3/dist/js/bootstrap.bundle.min.js"></script>
<script>
function updateFilenameAndPreview(input) {
    var fileName = "No file chosen";
    var container = document.getElementById('orig-preview-container');
    if (container) container.innerHTML = '';
    if (input.files && input.files.length > 0) {
        fileName = input.files[0].name;
        // Show preview
        if (container) {
            var reader = new FileReader();
            reader.onload = function(e) {
                container.innerHTML = '<div class="img-preview-label">Preview</div><img src="'+e.target.result+'" class="img-fluid img-preview">';
            }
            reader.readAsDataURL(input.files[0]);
        }
    }
    document.getElementById('file-filename').textContent = fileName;
}
function enableFileInput() {
    var fileInput = document.getElementById('image_file');
    var fileNameDiv = document.getElementById('file-filename');
    var previewDiv = document.getElementById('orig-preview-container');
    fileInput.disabled = false;
    fileInput.value = '';
    fileNameDiv.textContent = 'No file chosen';
    if (previewDiv) previewDiv.innerHTML = '';
}
document.addEventListener('DOMContentLoaded', function() {
    var fileInput = document.getElementById('image_file');
    var fileNameDiv = document.getElementById('file-filename');
    var previewDiv = document.getElementById('orig-preview-container');
    if (fileInput && fileNameDiv) {
        if (!fileInput.disabled) {
            fileInput.value = '';
            fileNameDiv.textContent = 'No file chosen';
            if (previewDiv) previewDiv.innerHTML = '';
        }
        fileInput.onchange = function() {
            updateFilenameAndPreview(this);
        };
        fileNameDiv.onclick = function() {
            if (!fileInput.disabled) fileInput.click();
        };
    }
});
</script>
</body>
</html>
