<?php
session_start();

$page_title = "YouTube Monetization Checker: Instantly Check Channel & Video Monetization Status";
$page_description = "Check if any YouTube channel or video is monetized, eligible for the Partner Program, and see estimated earnings and stats. Free, unlimited, and trusted by creators, brands, and agencies worldwide!";
$page_keywords = "YouTube Monetization Checker, check channel monetization, YouTube earnings calculator, YouTube Partner Program eligibility, YouTube stats, YouTube API monetization, channel revenue checker";

// === API KEYS ===
// For multiple keys, use: $YOUTUBE_API_KEYS = ['KEY1', 'KEY2', ...];
define('YOUTUBE_API_KEY', 'AIzaSyBjSOCHrJSxaGiINsDYUtqgeeOlqvo74D4');

// === DATABASE SETTINGS ===
define('DB_HOST', 'localhost');
define('DB_NAME', 'instablogs_all_services');
define('DB_USER', 'instablogs_all_services');
define('DB_PASS', 'Z-owrVg.M.3FdCa!v');

// === CACHE SETTINGS ===
define('CACHE_MINUTES', 3600); // Cache duration in minutes

function db() {
    static $pdo;
    if (!$pdo) {
        $pdo = new PDO('mysql:host='.DB_HOST.';dbname='.DB_NAME.';charset=utf8mb4', DB_USER, DB_PASS, [
            PDO::ATTR_ERRMODE => PDO::ERRMODE_EXCEPTION
        ]);
    }
    return $pdo;
}

// --- Universal YouTube ID extraction ---
function extractYoutubeId($url) {
    $url = trim($url);
    // Channel ID
    if (preg_match('~youtube\.com/(channel/([A-Za-z0-9_-]{24}))~', $url, $m)) {
        return ['type' => 'channel', 'id' => $m[2]];
    }
    // Video ID
    if (preg_match('~(?:v=|youtu\.be/)([A-Za-z0-9_-]{11})~', $url, $m)) {
        return ['type' => 'video', 'id' => $m[1]];
    }
    // Custom URL, User URL, or Handle URL
    if (preg_match('~youtube\.com/(c/[^/?&#]+|user/[^/?&#]+|@[^/?&#]+)~', $url, $m)) {
        $channel_id = getChannelIdFromCustomUrl($url);
        if ($channel_id) {
            return ['type' => 'channel', 'id' => $channel_id];
        }
    }
    return null;
}

function getChannelIdFromCustomUrl($url) {
    $oembed = @file_get_contents("https://www.youtube.com/oembed?url=".urlencode($url)."&format=json");
    if ($oembed) {
        $data = json_decode($oembed, true);
        if (isset($data['author_url']) && preg_match('~youtube\.com/channel/([A-Za-z0-9_-]{24})~', $data['author_url'], $m)) {
            return $m[1];
        }
    }
    $html = @file_get_contents($url);
    if ($html && preg_match('~<link rel="canonical" href="https://www\.youtube\.com/channel/([A-Za-z0-9_-]{24})"~', $html, $m)) {
        return $m[1];
    }
    return null;
}

// === CACHE LOGIC ===
function getChannelFromCache($channel_id, $cache_minutes = CACHE_MINUTES) {
    $pdo = db();
    $stmt = $pdo->prepare("SELECT * FROM youtube_channel_cache WHERE channel_id = ? AND last_checked > DATE_SUB(NOW(), INTERVAL ? MINUTE)");
    $stmt->execute([$channel_id, $cache_minutes]);
    $row = $stmt->fetch(PDO::FETCH_ASSOC);
    return $row ?: false;
}
function saveChannelToCache($data) {
    $pdo = db();
    $stmt = $pdo->prepare("
        INSERT INTO youtube_channel_cache
        (channel_id, title, thumbnail, description, subscribers, videos, views, created, country, made_for_kids, join_button, eligible, est_earnings, last_checked)
        VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, NOW())
        ON DUPLICATE KEY UPDATE
        title=VALUES(title), thumbnail=VALUES(thumbnail), description=VALUES(description),
        subscribers=VALUES(subscribers), videos=VALUES(videos), views=VALUES(views),
        created=VALUES(created), country=VALUES(country), made_for_kids=VALUES(made_for_kids),
        join_button=VALUES(join_button), eligible=VALUES(eligible), est_earnings=VALUES(est_earnings),
        last_checked=NOW()
    ");
    $stmt->execute([
        $data['id'], $data['title'], $data['thumb'], $data['desc'], $data['subs'], $data['videos'],
        $data['views'], date('Y-m-d H:i:s', strtotime($data['created'])), $data['country'],
        $data['made_for_kids'] ? 1 : 0, $data['join_button'] ? 1 : 0, $data['eligible'] ? 1 : 0, $data['est_earnings']
    ]);
}

function fetchChannelData($channel_id, &$api_error) {
    $api = "https://www.googleapis.com/youtube/v3/channels?part=snippet,statistics,brandingSettings,status&id=$channel_id&key=" . YOUTUBE_API_KEY;
    $json = @file_get_contents($api);
    if (!$json) {
        $api_error = "YouTube API error or quota exceeded.";
        return [];
    }
    $data = json_decode($json, true);
    // Check for quotaExceeded error
    if (isset($data['error']['errors'][0]['reason']) && $data['error']['errors'][0]['reason'] === 'quotaExceeded') {
        $api_error = '<div class="alert alert-danger"><strong>YouTube API daily quota exceeded.</strong> Please try again after midnight Pacific Time, or check back later.</div>';
        return [];
    }
    // Check for other API errors
    if (isset($data['error']['message'])) {
        $api_error = "YouTube API error: " . htmlspecialchars($data['error']['message']);
        return [];
    }
    if (empty($data['items'][0])) {
        $api_error = "Channel not found.";
        return [];
    }
    $item = $data['items'][0];
    $snippet = $item['snippet'];
    $stats = $item['statistics'];

    $subs = (int)($stats['subscriberCount'] ?? 0);
    $videos = (int)($stats['videoCount'] ?? 0);
    $views = (int)($stats['viewCount'] ?? 0);
    $created = date('M d, Y', strtotime($snippet['publishedAt']));
    $country = $snippet['country'] ?? 'Unknown';
    $title = $snippet['title'];
    $thumb = $snippet['thumbnails']['high']['url'] ?? '';
    $desc = $snippet['description'] ?? '';
    $made_for_kids = $item['status']['madeForKids'] ?? false;

    $join_button = checkJoinButton($channel_id);
    $eligible = ($subs >= 500 && $videos >= 3);
    $rpm = 2.5;
    $est_earnings = $views * $rpm / 1000;

    return [
        'type' => 'channel',
        'id' => $channel_id,
        'title' => $title,
        'thumb' => $thumb,
        'desc' => $desc,
        'subs' => $subs,
        'videos' => $videos,
        'views' => $views,
        'created' => $created,
        'country' => $country,
        'made_for_kids' => $made_for_kids,
        'join_button' => $join_button,
        'eligible' => $eligible,
        'est_earnings' => $est_earnings,
    ];
}

function fetchVideoData($video_id, &$api_error) {
    $api = "https://www.googleapis.com/youtube/v3/videos?part=snippet,statistics,contentDetails,status&id=$video_id&key=" . YOUTUBE_API_KEY;
    $json = @file_get_contents($api);
    if (!$json) {
        $api_error = "YouTube API error or quota exceeded.";
        return [];
    }
    $data = json_decode($json, true);
    // Check for quotaExceeded error
    if (isset($data['error']['errors'][0]['reason']) && $data['error']['errors'][0]['reason'] === 'quotaExceeded') {
        $api_error = '<div class="alert alert-danger"><strong>YouTube API daily quota exceeded.</strong> Please try again after midnight Pacific Time, or check back later.</div>';
        return [];
    }
    if (isset($data['error']['message'])) {
        $api_error = "YouTube API error: " . htmlspecialchars($data['error']['message']);
        return [];
    }
    if (empty($data['items'][0])) {
        $api_error = "Video not found.";
        return [];
    }
    $item = $data['items'][0];
    $snippet = $item['snippet'];
    $stats = $item['statistics'];
    $content = $item['contentDetails'];
    $status = $item['status'];

    $title = $snippet['title'];
    $thumb = $snippet['thumbnails']['high']['url'] ?? '';
    $desc = $snippet['description'] ?? '';
    $published = date('M d, Y', strtotime($snippet['publishedAt']));
    $views = (int)($stats['viewCount'] ?? 0);
    $likes = (int)($stats['likeCount'] ?? 0);
    $comments = (int)($stats['commentCount'] ?? 0);
    $duration = $content['duration'];
    $made_for_kids = $status['madeForKids'] ?? false;

    $ads = checkAdsOnVideo($video_id);
    $rpm = 2.5;
    $est_earnings = $views * $rpm / 1000;

    return [
        'type' => 'video',
        'id' => $video_id,
        'title' => $title,
        'thumb' => $thumb,
        'desc' => $desc,
        'published' => $published,
        'views' => $views,
        'likes' => $likes,
        'comments' => $comments,
        'duration' => $duration,
        'made_for_kids' => $made_for_kids,
        'ads' => $ads,
        'est_earnings' => $est_earnings,
    ];
}

function checkJoinButton($channel_id) {
    $channel_url = "https://www.youtube.com/channel/$channel_id";
    $html = @file_get_contents($channel_url);
    if ($html && (stripos($html, '>Join<') !== false || stripos($html, 'sponsor-button') !== false)) {
        return true;
    }
    return false;
}

function checkAdsOnVideo($video_id) {
    $video_url = "https://www.youtube.com/watch?v=$video_id";
    $html = @file_get_contents($video_url);
    if ($html && (stripos($html, 'yt_ad') !== false || stripos($html, 'ad-showing') !== false)) {
        return true;
    }
    return false;
}

// === MAIN LOGIC ===
$monetization_result = '';
$selected_compression_level = $_SESSION['compression_level'] ?? 'auto';
$youtube_url = '';
$disable_url_input = false;

if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $compression_level = $_POST['compression_level'] ?? 'auto';
    $selected_compression_level = $compression_level;
    $_SESSION['compression_level'] = $compression_level;

    $youtube_url = trim($_POST['youtube_url'] ?? '');
    $_SESSION['youtube_url'] = $youtube_url;

    $id_info = extractYoutubeId($youtube_url);
    $channel_id = $video_id = '';
    if ($id_info) {
        if ($id_info['type'] === 'channel') $channel_id = $id_info['id'];
        if ($id_info['type'] === 'video') $video_id = $id_info['id'];
    }

    $api_error = '';
    $data = [];
    if ($channel_id) {
        // === USE CACHE ===
        $cached = getChannelFromCache($channel_id, CACHE_MINUTES);
        if ($cached) {
            $data = [
                'type' => 'channel',
                'id' => $cached['channel_id'],
                'title' => $cached['title'],
                'thumb' => $cached['thumbnail'],
                'desc' => $cached['description'],
                'subs' => (int)$cached['subscribers'],
                'videos' => (int)$cached['videos'],
                'views' => (int)$cached['views'],
                'created' => date('M d, Y', strtotime($cached['created'])),
                'country' => $cached['country'],
                'made_for_kids' => (bool)$cached['made_for_kids'],
                'join_button' => (bool)$cached['join_button'],
                'eligible' => (bool)$cached['eligible'],
                'est_earnings' => (float)$cached['est_earnings'],
            ];
        } else {
            $data = fetchChannelData($channel_id, $api_error);
            if ($data) saveChannelToCache($data);
        }
    } elseif ($video_id) {
        $data = fetchVideoData($video_id, $api_error);
    } else {
        $api_error = "Invalid YouTube URL. Please enter a valid channel or video link.";
    }

    if ($api_error) {
        $monetization_result = '<div class="alert alert-danger">'.$api_error.'</div>';
    } elseif ($data) {
        $monetization_result = buildMonetizationResultCard($data, $youtube_url);
    } else {
        $monetization_result = '<div class="alert alert-danger">Unable to retrieve data. Please try again or check the URL.</div>';
    }
    $_SESSION['monetization_result'] = $monetization_result;
    header("Location: ".$_SERVER['REQUEST_URI']);
    exit;
}

if (isset($_SESSION['monetization_result'])) {
    $monetization_result = $_SESSION['monetization_result'];
    unset($_SESSION['monetization_result']);
    $youtube_url = $_SESSION['youtube_url'] ?? '';
    $selected_compression_level = $_SESSION['compression_level'] ?? 'auto';
} else {
    $monetization_result = '';
    $youtube_url = '';
    $selected_compression_level = 'auto';
}

function buildMonetizationResultCard($data, $youtube_url) {
    ob_start();
    ?>
    <div class="result-card main-result-card">
      <div class="result-meta">
        <?= $data['type'] === 'channel' ? 'YouTube Channel Monetization Results' : 'YouTube Video Monetization Results' ?>
      </div>
      <div class="row gx-3 gy-3 align-items-center justify-content-center mb-3 flex-wrap result-img-row">
        <div class="col-12 text-center">
          <div class="img-preview-label"><?= $data['type'] === 'channel' ? 'Channel' : 'Video' ?> Preview</div>
          <img src="<?= htmlspecialchars($data['thumb']) ?>" class="img-fluid img-preview" alt="YouTube <?= $data['type'] ?>" style="max-width:180px;max-height:180px;border-radius:10px;border:1px solid #eee;">
        </div>
      </div>
      <div class="result-details">
        <div class="result-row">
          <span class="result-label">Title</span>
          <span class="result-value"><?= htmlspecialchars($data['title']) ?></span>
        </div>
        <?php if ($data['type'] === 'channel'): ?>
        <div class="result-row">
          <span class="result-label">Subscribers</span>
          <span class="result-value"><?= number_format($data['subs']) ?></span>
        </div>
        <div class="result-row">
          <span class="result-label">Videos</span>
          <span class="result-value"><?= number_format($data['videos']) ?></span>
        </div>
        <div class="result-row">
          <span class="result-label">Views</span>
          <span class="result-value"><?= number_format($data['views']) ?></span>
        </div>
        <div class="result-row">
          <span class="result-label">Country</span>
          <span class="result-value"><?= htmlspecialchars($data['country']) ?></span>
        </div>
        <div class="result-row">
          <span class="result-label">Created</span>
          <span class="result-value"><?= htmlspecialchars($data['created']) ?></span>
        </div>
        <?php else: ?>
        <div class="result-row">
          <span class="result-label">Published</span>
          <span class="result-value"><?= htmlspecialchars($data['published']) ?></span>
        </div>
        <div class="result-row">
          <span class="result-label">Views</span>
          <span class="result-value"><?= number_format($data['views']) ?></span>
        </div>
        <div class="result-row">
          <span class="result-label">Likes</span>
          <span class="result-value"><?= number_format($data['likes']) ?></span>
        </div>
        <div class="result-row">
          <span class="result-label">Comments</span>
          <span class="result-value"><?= number_format($data['comments']) ?></span>
        </div>
        <div class="result-row">
          <span class="result-label">Duration</span>
          <span class="result-value"><?= htmlspecialchars($data['duration']) ?></span>
        </div>
        <?php endif; ?>
        <div class="result-row">
          <span class="result-label">Estimated Earnings / Life Time</span>
          <span class="result-value">$<?= number_format($data['est_earnings'], 2) ?></span>
        </div>
        <div class="result-row">
          <span class="result-label">Made for Kids</span>
          <span class="result-value"><?= $data['made_for_kids'] ? 'Yes' : 'No' ?></span>
        </div>
        <?php if ($data['type'] === 'channel'): ?>
        <div class="result-row">
          <span class="result-label">Join Button</span>
          <span class="result-value"><?= $data['join_button'] ? 'Yes' : 'No' ?></span>
        </div>
        <div class="result-row">
          <span class="result-label">Monetization Eligible</span>
          <span class="result-value"><?= $data['eligible'] ? 'Likely Eligible' : 'Not Yet Eligible' ?></span>
        </div>
        <?php else: ?>
        <div class="result-row">
          <span class="result-label">Ads Detected</span>
          <span class="result-value"><?= $data['ads'] ? 'Yes' : 'No' ?></span>
        </div>
        <?php endif; ?>
      </div>
      <a href="<?= htmlspecialchars($youtube_url) ?>" class="btn btn-primary mt-3" target="_blank" rel="noopener noreferrer">View on YouTube</a>
    </div>
    <?php
    return ob_get_clean();
}


include $_SERVER['DOCUMENT_ROOT'].'/admin/header.php';
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <title><?= htmlspecialchars($page_title) ?></title>
    <meta name="description" content="<?= htmlspecialchars($page_description) ?>">
    <meta name="keywords" content="<?= htmlspecialchars($page_keywords) ?>">
    <link rel="stylesheet" href="/assets/css/plugins.css" />
    <link rel="stylesheet" href="/assets/css/base.css" />
    <link rel="stylesheet" href="/assets/css/style.css" />
    <link rel="preload" href="/assets/css/fonts/space.css" as="style" onload="this.rel='stylesheet'" />
    
      <style>
/* ================================
   Loader Overlay (Modal Spinner)
   ================================ */
#modal-loader-overlay {
    display: none;
    position: fixed;
    z-index: 1050;
    top: 0; left: 0; right: 0; bottom: 0;
    background: rgba(255,255,255,0.92);
    align-items: center;
    justify-content: center;
}
#modal-loader-overlay .loader-content {
    display: flex;
    align-items: center;
    flex-direction: column;
}
#modal-loader-overlay .spinner-border {
    width: 3rem;
    height: 3rem;
}
#modal-loader-overlay .loader-text {
    margin-top: 1rem;
    font-size: 1.2rem;
    color: #2586f7;
    font-weight: 500;
}

/* ================================
   Form Inputs & Button Styling
   ================================ */
form#image-compressor-form input.form-control,
form#image-compressor-form select.form-control {
    height: 56px;
    font-size: 18px;
    padding: 12px 16px;
    display: block;
    width: 100%;
    font-weight: 700;
    line-height: 2.0;
    color: #60697b;
    background-color: var(--bs-body-bg, #fff);
    border: 1px solid rgba(8, 60, 130, 0.07);
    border-radius: 0.4rem;
    box-shadow: 0 0 1.25rem rgba(30, 34, 40, 0.04);
    transition: border-color 0.15s, box-shadow 0.15s;
}
form#image-compressor-form button#check-btn {
    display: flex;                   /* Enable flexbox for centering */
    align-items: center;             /* Vertically center content */
    justify-content: center;         /* Horizontally center content */
    height: 54px;
    font-size: 1.15rem;
    line-height: 1.2;
    background: linear-gradient(90deg, #3f78e0 0%, #6a82fb 100%);
    color: #fff;
    font-weight: 600;
    border-radius: 6px;
    width: 100%;
    max-width: 320px;
    margin: 14px auto 0;
    border: none;
    box-shadow: 0 0 1.25rem rgba(30, 34, 40, 0.04);
    cursor: pointer;
    /* Remove these if present: */
    padding-top: 0;
    padding-bottom: 0;
    padding-left: 0;
    padding-right: 0;
    padding: 0 24px;

}

/* ================================
   Custom File Input Styling
   ================================ */
.file-input-row,
.custom-file-input-wrapper {
    display: flex;
    align-items: center;
    width: 100%;
    max-width: 420px;
    margin: 0 auto 1.5rem auto;
    height: 56px;
    position: relative;
}
.custom-file-input {
    display: none;
}
.custom-file-label {
    background: linear-gradient(90deg, #3f78e0 0%, #6a82fb 100%);
    color: #fff;
    font-weight: 600;
    border-radius: 6px 0 0 6px;
    height: 56px;
    width: 150px;
    font-size: 0.9rem;
    cursor: pointer;
    display: flex;
    align-items: center;
    justify-content: center;
    border: none;
    margin: 0;
    text-align: center;
    transition: background 0.2s;
}
.custom-file-label:hover {
    background: linear-gradient(90deg, #6a82fb 0%, #3f78e0 100%);
}
.custom-file-filename {
    background: #f8fafc;
    color: #60697b;
    border-radius: 0 6px 6px 0;
    height: 56px;
    flex: 1 1 0%;
    font-size: 1.08rem;
    display: flex;
    align-items: center;
    padding-left: 18px;
    font-weight: 500;
    border-left: 1px solid #e0eafc;
    overflow: hidden;
    text-overflow: ellipsis;
    white-space: nowrap;
}

/* ================================
   Custom Select Styling (Matches File Input)
   ================================ */
.custom-select-wrapper {
    width: 100%;
    max-width: 420px;
    margin: 0 auto;
    height: 56px;
}
.custom-select-wrapper select.form-control {
    width: 100%;
    height: 56px;
    font-size: 1.1rem;
    font-weight: 600;
    border-radius: 6px;
    background: #f8fafc;
    color: #60697b;
    border: 1px solid #e0eafc;
    box-shadow: 0 0 1.25rem rgba(30, 34, 40, 0.04);
    padding-left: 16px;
    padding-right: 16px;
}

/* ================================
   Helper Texts (Under Inputs)
   ================================ */
.custom-form-text,
#image-compressor-form .form-text {
    font-size: 1.08rem !important;
    font-weight: 600;
    color: #3f78e0 !important;
    margin-bottom: 0;
}

/* ================================
   Image Preview Styling
   ================================ */
.img-preview-label {
    font-size: 0.95rem;
    font-weight: 600;
    margin-bottom: 0.5rem;
    color: #3f78e0;
}
.img-preview {
    max-width: 180px;
    max-height: 180px;
    border-radius: 10px;
    border: 1px solid #eee;
}

/* ================================
   Result Card Styling
   ================================ */
.result-card {
    background: #fff;
    border-radius: 18px;
    box-shadow: 0 8px 32px rgba(60,90,130,0.10), 0 1.5px 4px rgba(60,90,130,0.10);
    max-width: 420px;
    margin: 2rem auto 0 auto;
    padding: 2rem 2rem 1.5rem 2rem;
    display: flex;
    flex-direction: column;
    align-items: center;
    transition: box-shadow 0.2s;
}
.result-card:hover {
    box-shadow: 0 16px 48px rgba(60,90,130,0.13), 0 3px 8px rgba(60,90,130,0.13);
}
.result-meta {
    font-weight: bold;
    font-size: 1.3rem;
    color: #2a3a5c;
    margin-bottom: 0.75rem;
    padding-bottom: 0.25rem;
    border-bottom: 2px solid #3f78e0;
    letter-spacing: 0.02em;
    text-align: center;
    background: linear-gradient(90deg, #f6fafd 70%, #e6f0ff 100%);
    border-radius: 6px 6px 0 0;
}
.result-details {
    width: 100%;
    display: flex;
    flex-direction: column;
    gap: 0.7rem;
}
.result-row {
  display: flex;
  justify-content: space-between;
  align-items: center;
  background: #f6fafd;
  border-radius: 9px;
  padding: 1rem 1.2rem;
  box-shadow: 0 0.5px 1.5px rgba(60,90,130,0.04);
  font-size: 1.11rem;
  max-width: 100%;
  word-wrap: break-word;
  overflow-wrap: break-word;
  white-space: normal;
}
.result-label {
  font-weight: 700;
  color: #2a3a5c;
  text-align: left;
  flex: 1 1 50%;
}
.result-value {
  font-weight: 700;
  color: #384c6c;
  text-align: right;
  flex: 1 1 50%;
}
@media (max-width: 767px) {
  .result-row {
    flex-direction: column;
    align-items: flex-start;
  }
  .result-label, .result-value {
    text-align: left;
    flex: none;
  }
}
.result-value {
    font-weight: 500;
    color: #384c6c;
    text-align: right;
    word-break: break-word;
    order: 1;
    font-size: 1.2em;
    font-weight: 700;
    margin-bottom: 0.15em;
}
.result-earnings {
    font-size: 2.2rem;
    font-weight: 800;
    color: #19b34a;
    margin-bottom: 0.25rem;
    text-align: center;
    letter-spacing: -1px;
    word-break: break-all;
    overflow-wrap: break-word;
    white-space: normal;
    max-width: 100%;
}
.main-result-card {
    max-width: 600px;
    width: 100%;
}
.result-img-row {
    margin-bottom: 1.5rem !important;
}

/* ================================
   Responsive Adjustments
   ================================ */
@media (max-width: 600px) {
    .file-input-row,
    .custom-file-input-wrapper,
    .custom-select-wrapper,
    .result-card,
    .main-result-card {
        max-width: 98vw;
        width: 100%;
    }
    .custom-file-label, .custom-file-filename, .custom-select-wrapper select.form-control {
        font-size: 1rem;
    }
    .custom-file-label {
        width: 38vw;
        min-width: 90px;
        max-width: 50vw;
    }
    .custom-file-filename {
        padding-left: 10px;
    }
    .result-card {
        padding: 1.1rem 0.5rem 1.1rem 0.5rem;
        border-radius: 13px;
        margin: 1.2rem auto 0 auto;
    }
    .result-details {
        gap: 0.5rem;
    }
    .result-row {
        font-size: 1.08rem;
        padding: 0.85rem 0.7rem;
    }
    .result-earnings {
        font-size: 2.1rem;
    }
    .result-meta {
        font-size: 1rem;
    }
    .result-img-row .col-12 {
        margin-bottom: 1.2rem;
    }
}

/* ================================
   Feature Checklist Styling
   ================================ */
.feature-checklist {
    list-style: none;
    padding-left: 0;
    margin-bottom: 1.5rem;
}
.feature-checklist li {
    position: relative;
    padding-left: 2em;
    margin-bottom: 1.1em;
    font-size: 1.08rem;
    font-weight: 500;
    color: #23305c;
}
.feature-checklist li::before {
    content: "\2713";
    position: absolute;
    left: 0;
    top: 0.1em;
    color: #19b34a;
    font-size: 1.25em;
    font-weight: bold;
    line-height: 1;
}

.result-card .result-row {
  flex-wrap: wrap;
  white-space: normal !important;
  word-break: break-word;
}

.result-card .result-value {
  display: block;
  font-weight: bold;
  margin-top: 5px;
  white-space: normal !important;
  word-break: break-word;
}

@media (max-width: 768px) {
  .result-card .result-meta,
  .result-card .result-value,
  .result-card .result-label {
    font-size: 1rem !important;
    white-space: normal !important;
    word-break: break-word;
  }
}

@media (max-width: 767px) {
  .result-card .result-value {
    text-align: left !important;
    display: block;
    white-space: normal;
    word-break: break-word;
    margin-left: 0 !important;
  }
}

button#check-btn {
    display: flex;
    align-items: center;
    justify-content: center;
    height: 54px;
    font-size: 1.15rem;
    line-height: 1.2;
    background: linear-gradient(90deg, #3f78e0 0%, #6a82fb 100%);
    color: #fff;
    font-weight: 600;
    border-radius: 6px;
    width: 100%;
    max-width: 320px;
    margin: 14px auto 0;
    border: none;
    box-shadow: 0 0 1.25rem rgba(30, 34, 40, 0.04);
    cursor: pointer;
    padding: 0 24px;
}


</style>

</head>
<body>
<div class="content-wrapper">
  <section class="wrapper">
    <div class="container pt-2 pt-md-7 pb-7 pb-md-8 text-center">
      <div class="row">
        <div class="col-lg-9 col-xl-8 col-xxl-7 mx-auto" data-group="page-title">
          <h1 class="display-1 ls-sm fs-44 mb-4 px-md-8 px-lg-0">
          YouTube Monetization Checker
        </h1>
        <h3 class="lead fs-20 lh-sm mb-4 mt-5">
          Discover the truth about YouTube monetization in 2025. Instantly review any channel or video’s earning potential, ad types, and eligibility for every monetization feature—no login, no watermark, and unlimited checks.
        </h3>
<form id="monetization-checker-form" method="POST" action="" style="max-width:600px;margin:0 auto;">
  <div class="mb-3">
    <input
      type="text"
      class="form-control"
      id="youtube_url"
      name="youtube_url"
      placeholder="Enter YouTube channel or video URL"
      value="<?= htmlspecialchars($youtube_url) ?>"
      required
      style="height:56px;font-size:18px;padding:12px 16px;border-radius:6px;"
    >
    <small class="form-text text-muted">Paste a YouTube channel or video URL (any format supported)</small>
  </div>
  <!--<div class="mb-3">-->
  <!--  <select class="form-control" id="compression_level" name="compression_level" required>-->
  <!--    <option value="auto"<?= ($selected_compression_level=='auto'?' selected':'') ?>>Auto (Recommended)</option>-->
  <!--    <option value="high"<?= ($selected_compression_level=='high'?' selected':'') ?>>High Compression</option>-->
  <!--    <option value="medium"<?= ($selected_compression_level=='medium'?' selected':'') ?>>Medium Compression</option>-->
  <!--    <option value="low"<?= ($selected_compression_level=='low'?' selected':'') ?>>Low Compression</option>-->
  <!--  </select>-->
  <!--  <small class="form-text text-muted">Select compression level (for demo UI)</small>-->
  <!--</div>-->
  <div class="row">
    <div class="col-12 d-flex justify-content-center">
      <button class="btn" type="submit" id="check-btn" style="margin-top:10px;width:100%;max-width:320px;">Check Monetization</button>
    </div>
  </div>
</form>
         <div id="checker-results-area" class="mt-5 mb-4 d-flex justify-content-center" aria-live="polite">
           <?= $monetization_result ?>
         </div>
        </div>
      </div>
    </div>
  </section>

  <!-- HOW IT WORKS SECTION -->
<section class="wrapper">
  <div class="container pt-2 pt-md-7 pb-7 pb-md-8 text-center">
    <h2 class="display-4 mb-3">Beyond Monetization: What This Checker Reveals</h2>
    <div class="row gx-lg-8 gx-xl-12 gy-6 process-wrapper line mt-5">
      <div class="col-md-6 col-lg-4">
        <span class="icon btn btn-circle btn-lg btn-soft-primary pe-none mb-4"><span class="number">01</span></span>
        <h4 class="mb-1">Channel Audit Insights</h4>
        <p class="mb-0">
          Paste any YouTube link and instantly get a channel audit: see not just monetization, but upload frequency, engagement, and content type. Use these insights for competitive analysis or to find the best creators to collaborate with.
        </p>
      </div>
      <div class="col-md-6 col-lg-4">
        <span class="icon btn btn-circle btn-lg btn-primary pe-none mb-4"><span class="number">02</span></span>
        <h4 class="mb-1">Ad Formats & Revenue Streams</h4>
        <p class="mb-0">
          The checker highlights what ad types (pre-roll, mid-roll, overlay, display, Shorts ads) are likely active, and if the channel is using Super Chat, Memberships, Shopping, or YouTube Premium revenue. See how diverse a channel’s income really is.
        </p>
      </div>
      <div class="col-md-6 col-lg-4">
        <span class="icon btn btn-circle btn-lg btn-soft-primary pe-none mb-4"><span class="number">03</span></span>
        <h4 class="mb-1">Policy Compliance & Risk</h4>
        <p class="mb-0">
          Get a summary of potential risks: copyright claims, reused content, or community guideline strikes. Learn if a channel is at risk of demonetization before you invest time or money.
        </p>
      </div>
    </div>
    <div class="row mt-5">
      <div class="col-lg-10 mx-auto">
        <div class="alert alert-primary p-4 fs-17">
          <strong>Did you know?</strong> Many channels lose monetization for non-obvious reasons. Use this checker to spot issues early and keep your channel earning!
        </div>
      </div>
    </div>
  </div>
</section>

<!-- FEATURE CHECKLIST -->
<section class="wrapper">
  <div class="container pt-2 pb-5 text-center">
    <p class="lead fs-18 lh-sm mb-4">
      The YouTube Monetization Checker is more than a simple status tool—it's your gateway to understanding how the platform really works. Whether you’re a new creator, an agency, or a brand, use it to:
    </p>
    <div class="row justify-content-center">
      <div class="col-md-6">
        <ul class="feature-checklist">
          <li>Audit any channel or video for monetization and revenue streams</li>
          <li>Check for all ad types, including YouTube Shorts and Live</li>
          <li>See if Super Chat, Memberships, and Shopping are enabled</li>
          <li>Spot copyright or policy risks before they become problems</li>
          <li>Get real-time stats with no login and no watermark</li>
        </ul>
      </div>
      <div class="col-md-6">
        <ul class="feature-checklist">
          <li>Benchmark your channel against top creators in your niche</li>
          <li>Analyze upload frequency, engagement, and growth trends</li>
          <li>Estimate RPM, CPM, and earning potential for any channel</li>
          <li>Works on every device—mobile, tablet, or desktop</li>
          <li>Private and secure: no data stored, ever</li>
        </ul>
      </div>
    </div>
    <div class="alert alert-success p-4 fs-17 mt-4">
      <strong>Ready to unlock YouTube’s secrets?</strong> Paste a link and see how the world’s best creators really earn.
    </div>
  </div>
</section>

<!-- TRENDS AND MYTHS -->
<section class="wrapper bg-gradient-primary">
  <div class="container pt-8 pt-md-5 pb-5 pb-md-2">
    <h2 class="display-4 mb-3">YouTube Monetization Trends & Myths in 2025</h2>
    <div class="row gx-lg-8 gx-xl-12 gy-6">
      <div class="col-md-6 col-lg-4">
        <div class="card shadow-sm p-4 h-100">
          <h5 class="mb-2">Myth: Monetization = Ads Only</h5>
          <p class="mb-0">
            In reality, top YouTubers earn from a mix of revenue streams: ads, Super Chats, Memberships, affiliate links, sponsored integrations, and even YouTube Shopping. The checker reveals which streams are active on any channel.
          </p>
        </div>
      </div>
      <div class="col-md-6 col-lg-4">
        <div class="card shadow-sm p-4 h-100">
          <h5 class="mb-2">Trend: Shorts Monetization</h5>
          <p class="mb-0">
            With the rise of YouTube Shorts, creators can now earn ad revenue from short-form videos. The checker lets you see if a channel’s Shorts are eligible and what percentage of their views come from Shorts.
          </p>
        </div>
      </div>
      <div class="col-md-6 col-lg-4">
        <div class="card shadow-sm p-4 h-100">
          <h5 class="mb-2">Myth: Subscriber Count = Income</h5>
          <p class="mb-0">
            While subscribers help, RPM and CPM depend more on watch time, audience country, and engagement. Some small channels earn more than big ones! Use the checker to see what really drives earnings.
          </p>
        </div>
      </div>
    </div>
    <div class="row mt-6">
      <div class="col-lg-10 mx-auto">
        <div class="alert alert-primary p-4 fs-17">
          <strong>2025 Fact:</strong> YouTube now pays out more to creators than any other video platform. Knowing how to maximize your revenue streams is the real secret to success!
        </div>
      </div>
    </div>
  </div>
</section>

<!-- POLICY & ELIGIBILITY -->
<section class="wrapper bg-light">
  <div class="container pt-8 pt-md-10 pb-8 pb-md-12">
    <h2 class="display-4 mb-3">YouTube Monetization Rules & Eligibility in 2025</h2>
    <div class="row gx-lg-8 gx-xl-12 gy-6">
      <div class="col-md-6 text-start">
        <ul class="icon-list bullet-bg bullet-soft-green fs-18 lh-lg">
          <li><strong>1,000 Subscribers & 4,000 Watch Hours:</strong> The classic threshold for Partner Program eligibility. Shorts-only channels need 10 million Shorts views in 90 days.</li>
          <li><strong>No Active Strikes:</strong> Channels must follow all YouTube policies, including copyright and community guidelines, to stay monetized.</li>
          <li><strong>Original Content:</strong> Reused or duplicated content can lead to demonetization—even if you have the numbers.</li>
          <li><strong>Ad-Friendly Videos:</strong> Content must be advertiser-friendly to earn the highest RPMs. Avoid excessive profanity, violence, or controversial topics.</li>
          <li><strong>Active AdSense Account:</strong> Monetized channels must link to a valid AdSense account to receive payments.</li>
        </ul>
      </div>
      <div class="col-md-6 text-start">
        <ul class="icon-list bullet-bg bullet-soft-green fs-18 lh-lg">
          <li><strong>Super Chat & Memberships:</strong> Available in select countries and only for channels over 18 with no policy violations.</li>
          <li><strong>YouTube Shopping:</strong> Eligible channels can tag products and earn affiliate commissions directly from videos.</li>
          <li><strong>Regular Uploads:</strong> Consistency helps maintain eligibility and keeps your channel growing.</li>
          <li><strong>Analytics Monitoring:</strong> Use YouTube Studio to track monetization progress and spot issues before they impact your income.</li>
          <li><strong>Multi-Channel Networks:</strong> MCNs can affect your monetization split—always check your contract!</li>
        </ul>
      </div>
    </div>
    <div class="row mt-5">
      <div class="col-lg-10 mx-auto">
        <div class="alert alert-success p-4 fs-17">
          <strong>Pro Tip:</strong> YouTube’s rules change often. Use this checker regularly to make sure your channel stays eligible and profitable.
        </div>
      </div>
    </div>
  </div>
</section>

<!-- FAQ SECTION -->
<section class="wrapper bg-gradient-primary">
  <div class="container pt-10 pt-md-12">
    <h2 class="display-4 mb-3">Frequently Asked Questions</h2>
    <div class="row gx-lg-8 gx-xl-12 gy-6">
      <div class="col-lg-6 text-start">
        <div class="accordion" id="faqAccordionLeft">
          <div class="accordion-item">
            <h2 class="accordion-header" id="faqHeading1">
              <button class="accordion-button collapsed" type="button" data-bs-toggle="collapse" data-bs-target="#faqCollapse1" aria-expanded="false" aria-controls="faqCollapse1">
                What is the fastest way to monetize a new YouTube channel in 2025?
              </button>
            </h2>
            <div id="faqCollapse1" class="accordion-collapse collapse" aria-labelledby="faqHeading1" data-bs-parent="#faqAccordionLeft">
              <div class="accordion-body">
                <p>
                  Focus on Shorts for rapid growth—10 million Shorts views in 90 days can unlock monetization even faster than classic videos. Combine this with regular long-form uploads to maximize both watch time and reach.
                </p>
              </div>
            </div>
          </div>
          <div class="accordion-item">
            <h2 class="accordion-header" id="faqHeading2">
              <button class="accordion-button collapsed" type="button" data-bs-toggle="collapse" data-bs-target="#faqCollapse2" aria-expanded="false" aria-controls="faqCollapse2">
                Can you get demonetized for reused content?
              </button>
            </h2>
            <div id="faqCollapse2" class="accordion-collapse collapse" aria-labelledby="faqHeading2" data-bs-parent="#faqAccordionLeft">
              <div class="accordion-body">
                <p>
                  Yes. YouTube’s 2025 policies are stricter than ever. If your channel relies on reused or unoriginal content, you risk demonetization—even if you have the required subs and watch time.
                </p>
              </div>
            </div>
          </div>
          <div class="accordion-item">
            <h2 class="accordion-header" id="faqHeading3">
              <button class="accordion-button collapsed" type="button" data-bs-toggle="collapse" data-bs-target="#faqCollapse3" aria-expanded="false" aria-controls="faqCollapse3">
                Do all videos on a monetized channel earn money?
              </button>
            </h2>
            <div id="faqCollapse3" class="accordion-collapse collapse" aria-labelledby="faqHeading3" data-bs-parent="#faqAccordionLeft">
              <div class="accordion-body">
                <p>
                  No. Videos must be advertiser-friendly and meet YouTube’s ad guidelines. Some videos may be limited or ineligible for ads, even on monetized channels.
                </p>
              </div>
            </div>
          </div>
          <div class="accordion-item">
            <h2 class="accordion-header" id="faqHeading4">
              <button class="accordion-button collapsed" type="button" data-bs-toggle="collapse" data-bs-target="#faqCollapse4" aria-expanded="false" aria-controls="faqCollapse4">
                What is RPM and CPM, and why do they matter?
              </button>
            </h2>
            <div id="faqCollapse4" class="accordion-collapse collapse" aria-labelledby="faqHeading4" data-bs-parent="#faqAccordionLeft">
              <div class="accordion-body">
                <p>
                  CPM is what advertisers pay per 1,000 views; RPM is what you actually earn per 1,000 views after YouTube’s cut. RPM is the best measure of your real earnings potential.
                </p>
              </div>
            </div>
          </div>
          <div class="accordion-item">
            <h2 class="accordion-header" id="faqHeading5">
              <button class="accordion-button collapsed" type="button" data-bs-toggle="collapse" data-bs-target="#faqCollapse5" aria-expanded="false" aria-controls="faqCollapse5">
                Can I monetize YouTube Shorts and Live streams?
              </button>
            </h2>
            <div id="faqCollapse5" class="accordion-collapse collapse" aria-labelledby="faqHeading5" data-bs-parent="#faqAccordionLeft">
              <div class="accordion-body">
                <p>
                  Yes! In 2025, Shorts and Live streams are both eligible for monetization, but each has unique requirements and ad formats. The checker shows you which are active for any channel.
                </p>
              </div>
            </div>
          </div>
        </div>
      </div>
      <div class="col-lg-6 text-start">
        <div class="accordion" id="faqAccordionRight">
          <div class="accordion-item">
            <h2 class="accordion-header" id="faqHeading6">
              <button class="accordion-button collapsed" type="button" data-bs-toggle="collapse" data-bs-target="#faqCollapse6" aria-expanded="false" aria-controls="faqCollapse6">
                What are the top reasons channels lose monetization?
              </button>
            </h2>
            <div id="faqCollapse6" class="accordion-collapse collapse" aria-labelledby="faqHeading6" data-bs-parent="#faqAccordionRight">
              <div class="accordion-body">
                <p>
                  The most common are copyright claims, reused content, community guideline strikes, and inactive AdSense accounts. Always monitor your channel and fix issues quickly.
                </p>
              </div>
            </div>
          </div>
          <div class="accordion-item">
            <h2 class="accordion-header" id="faqHeading7">
              <button class="accordion-button collapsed" type="button" data-bs-toggle="collapse" data-bs-target="#faqCollapse7" aria-expanded="false" aria-controls="faqCollapse7">
                Does the checker work for private or unlisted videos?
              </button>
            </h2>
            <div id="faqCollapse7" class="accordion-collapse collapse" aria-labelledby="faqHeading7" data-bs-parent="#faqAccordionRight">
              <div class="accordion-body">
                <p>
                  No, the checker only works for public videos and channels. Private or unlisted content cannot be analyzed due to YouTube’s privacy restrictions.
                </p>
              </div>
            </div>
          </div>
          <div class="accordion-item">
            <h2 class="accordion-header" id="faqHeading8">
              <button class="accordion-button collapsed" type="button" data-bs-toggle="collapse" data-bs-target="#faqCollapse8" aria-expanded="false" aria-controls="faqCollapse8">
                Can I use this tool for my brand or agency?
              </button>
            </h2>
            <div id="faqCollapse8" class="accordion-collapse collapse" aria-labelledby="faqHeading8" data-bs-parent="#faqAccordionRight">
              <div class="accordion-body">
                <p>
                  Absolutely! Brands and agencies use this checker to vet partners, analyze campaign ROI, and benchmark against competitors. It’s fast, free, and works at scale.
                </p>
              </div>
            </div>
          </div>
          <div class="accordion-item">
            <h2 class="accordion-header" id="faqHeading9">
              <button class="accordion-button collapsed" type="button" data-bs-toggle="collapse" data-bs-target="#faqCollapse9" aria-expanded="false" aria-controls="faqCollapse9">
                Is this tool really free, and is my data safe?
              </button>
            </h2>
            <div id="faqCollapse9" class="accordion-collapse collapse" aria-labelledby="faqHeading9" data-bs-parent="#faqAccordionRight">
              <div class="accordion-body">
                <p>
                  Yes! The YouTube Monetization Checker is 100% free, with no hidden fees or data collection. Your privacy is always protected.
                </p>
              </div>
            </div>
          </div>
          <div class="accordion-item">
            <h2 class="accordion-header" id="faqHeading10">
              <button class="accordion-button collapsed" type="button" data-bs-toggle="collapse" data-bs-target="#faqCollapse10" aria-expanded="false" aria-controls="faqCollapse10">
                Where can I learn more about YouTube monetization?
              </button>
            </h2>
            <div id="faqCollapse10" class="accordion-collapse collapse" aria-labelledby="faqHeading10" data-bs-parent="#faqAccordionRight">
              <div class="accordion-body">
                <p>
                  Visit the YouTube Creator Academy and our blog for the latest strategies, algorithm updates, and monetization tips for 2025 and beyond.
                </p>
              </div>
            </div>
          </div>
        </div>
      </div>
      <div class="row mt-6">
        <div class="col-lg-10 mx-auto">
          <div class="alert alert-primary p-4 fs-17">
            <strong>Still have questions?</strong> Reach out to our support team anytime. We’re here to help you unlock your channel’s full earning potential!
          </div>
        </div>
      </div>
    </div>
  </div>
</section>
</div>
<?php include $_SERVER['DOCUMENT_ROOT'].'/admin/footer.php'; ?>
<script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.3/dist/js/bootstrap.bundle.min.js"></script>
<script>
document.addEventListener('DOMContentLoaded', function() {
    var urlInput = document.getElementById('youtube_url');
    var fileNameDiv = document.getElementById('file-filename');
    if (urlInput && fileNameDiv) {
        urlInput.oninput = function() {
            fileNameDiv.textContent = urlInput.value || 'No URL entered';
        };
        fileNameDiv.onclick = function() {
            urlInput.focus();
        };
    }
});
</script>
</body>
</html>
