<?php
session_start();

$compressed_dir = __DIR__ . '/svg-compressed/';
$compressed_url = '/tools/compress-image/svg-compressed/';
if (!is_dir($compressed_dir)) mkdir($compressed_dir, 0777, true);

$compression_result = '';
$compression_level_label = '';
$selected_compression_level = 'auto';
$disable_file_input = false;
$orig_img_url = '';
$orig_size = 0;
$orig_ext = '';
$orig_mime = '';

// Helper: Human-readable compression level
function getCompressionLevelLabel($level) {
    switch ($level) {
        case 'high': return 'High';
        case 'medium': return 'Medium';
        case 'low': return 'Low';
        default: return 'Auto (Recommended)';
    }
}

// Helper: Compress SVG using SVGO or manual optimization
function compress_svg($input, $output) {
    $which = trim(shell_exec("which svgo"));
    if ($which) {
        $cmd = "svgo ".escapeshellarg($input)." --output ".escapeshellarg($output);
        exec($cmd, $out, $ret);
        if ($ret === 0 && file_exists($output)) return true;
    }
    $svg_content = file_get_contents($input);
    if ($svg_content === false) return false;
    $svg_content = preg_replace('/>\s+</', '><', $svg_content);
    $svg_content = preg_replace('/<!--.*?-->/', '', $svg_content);
    $svg_content = preg_replace('/\s+/', ' ', $svg_content);
    $svg_content = trim($svg_content);
    $svg_content = preg_replace('/\s*(xmlns:[^=]*="[^"]*")\s*/', '', $svg_content);
    $svg_content = preg_replace('/\s*(id="[^"]*")\s*/', '', $svg_content);
    $svg_content = preg_replace('/\s*(class="[^"]*")\s*/', '', $svg_content);
    return file_put_contents($output, $svg_content) !== false;
}

// --- Download handler: serve file if ?download=filename is set ---
if (isset($_GET['download'])) {
    $file = basename($_GET['download']);
    if (!preg_match('/^svgcmp_[a-zA-Z0-9]+\.svg$/', $file)) {
        http_response_code(400);
        die('Invalid file name.');
    }
    $filepath = $compressed_dir . $file;
    if (!file_exists($filepath)) {
        http_response_code(404);
        die('File not found.');
    }
    header('Content-Description: File Transfer');
    header('Content-Type: image/svg+xml');
    header('Content-Disposition: attachment; filename="' . $file . '"');
    header('Expires: 0');
    header('Cache-Control: must-revalidate');
    header('Pragma: public');
    header('Content-Length: ' . filesize($filepath));
    flush();
    readfile($filepath);
    exit;
}

// --- HANDLE POST: Compress and Redirect ---
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $max_size = 6 * 1024 * 1024; // 6MB
    $compression_level = $_POST['compression_level'] ?? 'auto';
    $compression_level_label = getCompressionLevelLabel($compression_level);
    $selected_compression_level = $compression_level;
    $success = false;
    $unique_id = uniqid('svgcmp_');
    $download_name = '';
    $compressed_file_path = '';
    $compressed_img_url = '';
    $orig_size = 0;

    // If new file uploaded, handle upload and store path in session
    if (isset($_FILES['image_file']) && $_FILES['image_file']['error'] === UPLOAD_ERR_OK) {
        $file = $_FILES['image_file'];
        if ($file['size'] > $max_size) {
            $_SESSION['compression_result'] = '<div class="alert alert-danger">File too large. Maximum size is 6MB.</div>';
        } else {
            $mime = mime_content_type($file['tmp_name']);
            if ($mime === 'image/svg+xml' || strtolower(pathinfo($file['name'], PATHINFO_EXTENSION)) === 'svg') {
                $orig_ext = '.svg';
                $orig_name = $unique_id . '_orig' . $orig_ext;
                $orig_path = $compressed_dir . $orig_name;
                move_uploaded_file($file['tmp_name'], $orig_path);
                $_SESSION['uploaded_svg_path'] = $orig_path;
                $_SESSION['uploaded_svg_url']  = $compressed_url . $orig_name;
                $_SESSION['uploaded_svg_size'] = filesize($orig_path);
                $_SESSION['uploaded_svg_mime'] = $mime;
                $orig_img_url = $_SESSION['uploaded_svg_url'];
                $orig_size = $_SESSION['uploaded_svg_size'];
                $orig_mime = $mime;
                $disable_file_input = true;
            } else {
                $_SESSION['compression_result'] = '<div class="alert alert-danger">Only SVG images are supported for this tool.</div>';
            }
        }
    }
    // If user changed compression level after upload, reuse previous file
    elseif (isset($_SESSION['uploaded_svg_path']) && file_exists($_SESSION['uploaded_svg_path'])) {
        $orig_path = $_SESSION['uploaded_svg_path'];
        $orig_img_url = $_SESSION['uploaded_svg_url'];
        $orig_size = $_SESSION['uploaded_svg_size'];
        $orig_mime = $_SESSION['uploaded_svg_mime'];
        $orig_ext = '.svg';
        $disable_file_input = true;
    }

    // If we have a valid original file, proceed to compress
    if (!empty($orig_path) && file_exists($orig_path)) {
        $download_name = $unique_id . '.svg';
        $compressed_file_path = $compressed_dir . $download_name;
        $success = compress_svg($orig_path, $compressed_file_path);
    }

    // Handle results
    if ($success && file_exists($compressed_file_path)) {
        $compressed_size = filesize($compressed_file_path);
        $compressed_img_url = $compressed_url . $download_name;
        if ($compressed_size >= $orig_size) {
            $_SESSION['compression_result'] = '
            <div class="result-card main-result-card">
              <div class="result-meta">SVG Compression Results</div>
              <div class="row gx-3 gy-3 align-items-center justify-content-center mb-3 flex-wrap result-img-row">
                <div class="col-12 col-md-6 text-center mb-2 mb-md-0">
                  <div class="img-preview-label">Original</div>
                  <img src="'.htmlspecialchars($orig_img_url).'" class="img-fluid img-preview" alt="Original SVG" style="max-width:180px;max-height:180px;border-radius:10px;border:1px solid #eee;">
                </div>
                <div class="col-12 col-md-6 text-center">
                  <div class="img-preview-label">Original (Optimized)</div>
                  <img src="'.htmlspecialchars($orig_img_url).'" class="img-fluid img-preview" alt="Original SVG" style="max-width:180px;max-height:180px;border-radius:10px;border:1px solid #eee;">
                </div>
              </div>
              <div class="result-details">
                <div class="result-row flex-column flex-md-row text-center text-md-start">
                  <span class="result-label">Compression Level</span>
                  <span class="result-value">'.htmlspecialchars($compression_level_label).'</span>
                </div>
                <div class="result-row flex-column flex-md-row text-center text-md-start">
                  <span class="result-label">Original Size</span>
                  <span class="result-value">'.number_format($orig_size/1024, 2).' KB</span>
                </div>
                <div class="result-row flex-column flex-md-row text-center text-md-start">
                  <span class="result-label">Final Size</span>
                  <span class="result-value">'.number_format($orig_size/1024, 2).' KB</span>
                </div>
                <div class="result-row flex-column flex-md-row text-center text-md-start">
                  <span class="result-label">Reduction</span>
                  <span class="result-value">0%</span>
                </div>
              </div>
              <div class="alert alert-info mt-3 mb-0">
                <strong>No further compression possible.</strong> The image is already optimized.
              </div>
              <a href="?download='.urlencode(basename($orig_path)).'" class="btn btn-primary mt-3">Download Original SVG</a>
            </div>';
        } else {
            $percent = $orig_size > 0 ? round(100 - ($compressed_size / $orig_size * 100), 1) : 0;
            $_SESSION['compression_result'] = '
            <div class="result-card main-result-card">
              <div class="result-meta">SVG Compression Results</div>
              <div class="row gx-3 gy-3 align-items-center justify-content-center mb-3 flex-wrap result-img-row">
                <div class="col-12 col-md-6 text-center mb-2 mb-md-0">
                  <div class="img-preview-label">Original</div>
                  <img src="'.htmlspecialchars($orig_img_url).'" class="img-fluid img-preview" alt="Original SVG" style="max-width:180px;max-height:180px;border-radius:10px;border:1px solid #eee;">
                </div>
                <div class="col-12 col-md-6 text-center">
                  <div class="img-preview-label">Compressed</div>
                  <img src="'.htmlspecialchars($compressed_img_url).'" class="img-fluid img-preview" alt="Compressed SVG" style="max-width:180px;max-height:180px;border-radius:10px;border:1px solid #eee;">
                </div>
              </div>
              <div class="result-details">
                <div class="result-row flex-column flex-md-row text-center text-md-start">
                  <span class="result-label">Compression Level</span>
                  <span class="result-value">'.htmlspecialchars($compression_level_label).'</span>
                </div>
                <div class="result-row flex-column flex-md-row text-center text-md-start">
                  <span class="result-label">Original Size</span>
                  <span class="result-value">'.number_format($orig_size/1024, 2).' KB</span>
                </div>
                <div class="result-row flex-column flex-md-row text-center text-md-start">
                  <span class="result-label">Compressed Size</span>
                  <span class="result-value">'.number_format($compressed_size/1024, 2).' KB</span>
                </div>
                <div class="result-row flex-column flex-md-row text-center text-md-start">
                  <span class="result-label">Reduction</span>
                  <span class="result-value">'.$percent.'%</span>
                </div>
              </div>
              <a href="?download='.urlencode($download_name).'" class="btn btn-primary mt-3">Download Compressed SVG</a>
            </div>';
        }
    } elseif (!isset($_SESSION['compression_result'])) {
        $_SESSION['compression_result'] = '<div class="alert alert-danger">Compression failed. Please try another SVG image.</div>';
    }
    header("Location: ".$_SERVER['REQUEST_URI']);
    exit;
}

// On GET, show results only if they exist in session, then clear them
if (isset($_SESSION['compression_result'])) {
    $compression_result = $_SESSION['compression_result'];
    unset($_SESSION['compression_result']);
    if (isset($_SESSION['uploaded_svg_url'])) {
        $orig_img_url = $_SESSION['uploaded_svg_url'];
        $disable_file_input = true;
    }
    if (isset($_SESSION['compression_level'])) {
        $selected_compression_level = $_SESSION['compression_level'];
    }
} else {
    $compression_result = '';
    $disable_file_input = false;
    $orig_img_url = '';
}

// SEO Meta
$page_title = "SVG Compressor – Compress SVG Files Online Fast & Free";
$page_description = "Compress SVG images instantly online. Reduce SVG file size while preserving crisp quality and code. No watermark, unlimited use, works on any device. The best SVG optimizer for web, design, and more.";
$page_keywords = "SVG Compressor, compress SVG, SVG optimization, reduce SVG size, online SVG compressor, free SVG tool, lossless SVG compression, vector image optimizer, web image optimization, fast SVG compressor";

include $_SERVER['DOCUMENT_ROOT'].'/admin/header.php';
?>
<!DOCTYPE html>
<html>
<head>
    <title><?= htmlspecialchars($page_title) ?></title>
    <meta name="description" content="<?= htmlspecialchars($page_description) ?>">
    <meta name="keywords" content="<?= htmlspecialchars($page_keywords) ?>">
    <link rel="stylesheet" href="/assets/css/plugins.css" />
    <link rel="stylesheet" href="/assets/css/base.css" />
    <link rel="stylesheet" href="/assets/css/style.css" />
    <link rel="preload" href="/assets/css/fonts/space.css" as="style" onload="this.rel='stylesheet'" />
<style>
    
    /* ================================
   Loader Overlay (Modal Spinner)
   ================================ */
#modal-loader-overlay {
    display: none;
    position: fixed;
    z-index: 1050;
    top: 0; left: 0; right: 0; bottom: 0;
    background: rgba(255,255,255,0.92);
    align-items: center;
    justify-content: center;
}
#modal-loader-overlay .loader-content {
    display: flex;
    align-items: center;
    flex-direction: column;
}
#modal-loader-overlay .spinner-border {
    width: 3rem;
    height: 3rem;
}
#modal-loader-overlay .loader-text {
    margin-top: 1rem;
    font-size: 1.2rem;
    color: #2586f7;
    font-weight: 500;
}

/* ================================
   Form Inputs & Button Styling
   ================================ */
form#image-compressor-form input.form-control,
form#image-compressor-form select.form-control {
    height: 56px;
    font-size: 18px;
    padding: 12px 16px;
    display: block;
    width: 100%;
    font-weight: 700;
    line-height: 2.0;
    color: #60697b;
    background-color: var(--bs-body-bg, #fff);
    border: 1px solid rgba(8, 60, 130, 0.07);
    border-radius: 0.4rem;
    box-shadow: 0 0 1.25rem rgba(30, 34, 40, 0.04);
    transition: border-color 0.15s, box-shadow 0.15s;
}
form#image-compressor-form button#compress-btn {
    display: flex;                   /* Enable flexbox for centering */
    align-items: center;             /* Vertically center content */
    justify-content: center;         /* Horizontally center content */
    height: 54px;
    font-size: 1.15rem;
    line-height: 1.2;
    background: linear-gradient(90deg, #3f78e0 0%, #6a82fb 100%);
    color: #fff;
    font-weight: 600;
    border-radius: 6px;
    width: 100%;
    max-width: 320px;
    margin: 14px auto 0;
    border: none;
    box-shadow: 0 0 1.25rem rgba(30, 34, 40, 0.04);
    cursor: pointer;
    /* Remove these if present: */
    padding-top: 0;
    padding-bottom: 0;
    padding-left: 0;
    padding-right: 0;
    padding: 0 24px;

}

/* ================================
   Custom File Input Styling
   ================================ */
.file-input-row,
.custom-file-input-wrapper {
    display: flex;
    align-items: center;
    width: 100%;
    max-width: 420px;
    margin: 0 auto 1.5rem auto;
    height: 56px;
    position: relative;
}
.custom-file-input {
    display: none;
}
.custom-file-label {
    background: linear-gradient(90deg, #3f78e0 0%, #6a82fb 100%);
    color: #fff;
    font-weight: 600;
    border-radius: 6px 0 0 6px;
    height: 56px;
    width: 160px;
    font-size: .90rem;
    cursor: pointer;
    display: flex;
    align-items: center;
    justify-content: center;
    border: none;
    margin: 0;
    text-align: center;
    transition: background 0.2s;
}
.custom-file-label:hover {
    background: linear-gradient(90deg, #6a82fb 0%, #3f78e0 100%);
}
.custom-file-filename {
    background: #f8fafc;
    color: #60697b;
    border-radius: 0 6px 6px 0;
    height: 56px;
    flex: 1 1 0%;
    font-size: 1.08rem;
    display: flex;
    align-items: center;
    padding-left: 18px;
    font-weight: 500;
    border-left: 1px solid #e0eafc;
    overflow: hidden;
    text-overflow: ellipsis;
    white-space: nowrap;
}

/* ================================
   Custom Select Styling (Matches File Input)
   ================================ */
.custom-select-wrapper {
    width: 100%;
    max-width: 420px;
    margin: 0 auto;
    height: 56px;
}
.custom-select-wrapper select.form-control {
    width: 100%;
    height: 56px;
    font-size: 1.1rem;
    font-weight: 600;
    border-radius: 6px;
    background: #f8fafc;
    color: #60697b;
    border: 1px solid #e0eafc;
    box-shadow: 0 0 1.25rem rgba(30, 34, 40, 0.04);
    padding-left: 16px;
    padding-right: 16px;
}

/* ================================
   Helper Texts (Under Inputs)
   ================================ */
.custom-form-text,
#image-compressor-form .form-text {
    font-size: 1.08rem !important;
    font-weight: 600;
    color: #3f78e0 !important;
    margin-bottom: 0;
}

/* ================================
   Image Preview Styling
   ================================ */
.img-preview-label {
    font-size: 0.95rem;
    font-weight: 600;
    margin-bottom: 0.5rem;
    color: #3f78e0;
}
.img-preview {
    max-width: 180px;
    max-height: 180px;
    border-radius: 10px;
    border: 1px solid #eee;
}

/* ================================
   Result Card Styling
   ================================ */
.result-card {
    background: #fff;
    border-radius: 18px;
    box-shadow: 0 8px 32px rgba(60,90,130,0.10), 0 1.5px 4px rgba(60,90,130,0.10);
    max-width: 420px;
    margin: 2rem auto 0 auto;
    padding: 2rem 2rem 1.5rem 2rem;
    display: flex;
    flex-direction: column;
    align-items: center;
    transition: box-shadow 0.2s;
}
.result-card:hover {
    box-shadow: 0 16px 48px rgba(60,90,130,0.13), 0 3px 8px rgba(60,90,130,0.13);
}
.result-meta {
    font-weight: bold;
    font-size: 1.3rem;
    color: #2a3a5c;
    margin-bottom: 0.75rem;
    padding-bottom: 0.25rem;
    border-bottom: 2px solid #3f78e0;
    letter-spacing: 0.02em;
    text-align: center;
    background: linear-gradient(90deg, #f6fafd 70%, #e6f0ff 100%);
    border-radius: 6px 6px 0 0;
}
.result-details {
    width: 100%;
    display: flex;
    flex-direction: column;
    gap: 0.7rem;
}
.result-row {
    display: flex;
    justify-content: space-between;
    align-items: center;
    background: #f6fafd;
    border-radius: 9px;
    padding: 1rem 1.2rem;
    box-shadow: 0 0.5px 1.5px rgba(60,90,130,0.04);
    font-size: 1.11rem;
    max-width: 100%;
    word-wrap: break-word;
    overflow-wrap: break-word;
    white-space: normal;
    flex-direction: column;
    align-items: flex-start;
    text-align: left;
}
.result-label {
    font-weight: 700;
    color: #2a3a5c;
    order: 2;
    margin-top: 0.1em;
    font-size: 1em;
}
.result-value {
    font-weight: 500;
    color: #384c6c;
    text-align: right;
    word-break: break-word;
    order: 1;
    font-size: 1.2em;
    font-weight: 700;
    margin-bottom: 0.15em;
}
.result-earnings {
    font-size: 2.2rem;
    font-weight: 800;
    color: #19b34a;
    margin-bottom: 0.25rem;
    text-align: center;
    letter-spacing: -1px;
    word-break: break-all;
    overflow-wrap: break-word;
    white-space: normal;
    max-width: 100%;
}
.main-result-card {
    max-width: 600px;
    width: 100%;
}
.result-img-row {
    margin-bottom: 1.5rem !important;
}

/* ================================
   Responsive Adjustments
   ================================ */
@media (max-width: 600px) {
    .file-input-row,
    .custom-file-input-wrapper,
    .custom-select-wrapper,
    .result-card,
    .main-result-card {
        max-width: 98vw;
        width: 100%;
    }
    .custom-file-label, .custom-file-filename, .custom-select-wrapper select.form-control {
        font-size: 1rem;
    }
    .custom-file-label {
        width: 38vw;
        min-width: 90px;
        max-width: 50vw;
    }
    .custom-file-filename {
        padding-left: 10px;
    }
    .result-card {
        padding: 1.1rem 0.5rem 1.1rem 0.5rem;
        border-radius: 13px;
        margin: 1.2rem auto 0 auto;
    }
    .result-details {
        gap: 0.5rem;
    }
    .result-row {
        font-size: 1.08rem;
        padding: 0.85rem 0.7rem;
    }
    .result-earnings {
        font-size: 2.1rem;
    }
    .result-meta {
        font-size: 1rem;
    }
    .result-img-row .col-12 {
        margin-bottom: 1.2rem;
    }
}

/* ================================
   Feature Checklist Styling
   ================================ */
.feature-checklist {
    list-style: none;
    padding-left: 0;
    margin-bottom: 1.5rem;
}
.feature-checklist li {
    position: relative;
    padding-left: 2em;
    margin-bottom: 1.1em;
    font-size: 1.08rem;
    font-weight: 500;
    color: #23305c;
}
.feature-checklist li::before {
    content: "\2713";
    position: absolute;
    left: 0;
    top: 0.1em;
    color: #19b34a;
    font-size: 1.25em;
    font-weight: bold;
    line-height: 1;
}
  .custom-file-filename {
      cursor: pointer;
      user-select: none;
    }
    
</style>
</head>
<body>
<div class="content-wrapper">

  <!-- Loader Overlay -->
  <div id="modal-loader-overlay">
    <div class="loader-content">
      <div class="spinner-border text-primary" role="status">
        <span class="visually-hidden">Loading...</span>
      </div>
      <span class="loader-text">Compressing SVG Image...</span>
    </div>
  </div>

  <!-- HERO + COMPRESSOR FORM -->
  <section class="wrapper">
    <div class="container pt-2 pt-md-7 pb-7 pb-md-8 text-center">
      <div class="row">
        <div class="col-lg-9 col-xl-8 col-xxl-7 mx-auto" data-group="page-title">
         <h1 class="display-1 ls-sm fs-44 mb-4 px-md-8 px-lg-0">
           SVG Image Compressor <span class="underline-3 style-1 primary"></span>
         </h1>
         <h3 class="lead fs-20 lh-sm mb-4 mt-5">
           Compress SVG Image files online in seconds. Shrink vector graphics, icons, and illustrations while keeping perfect quality and code. No watermark, unlimited use, works on any device.
         </h3>
<form id="image-compressor-form" method="POST" enctype="multipart/form-data" action="" style="max-width:600px;margin:0 auto;">
  <div class="mb-3">
    <div class="custom-file-input-wrapper file-input-row">
      <input
        type="file"
        class="custom-file-input"
        id="image_file"
        name="image_file"
        accept="image/svg+xml,.svg"
        <?= $disable_file_input ? 'disabled' : '' ?>
        required
        onchange="updateFilenameAndPreview(this)"
      >
      <label for="image_file" class="custom-file-label" id="choose-file-label">Choose SVG</label>
      <div class="custom-file-filename" id="file-filename" onclick="if(!document.getElementById('image_file').disabled){document.getElementById('image_file').click();}">No file chosen</div>
    </div>
    <small class="form-text text-muted">Choose an SVG file (max 6MB)</small>
    <div id="orig-preview-container" class="mt-3">
      <?php if ($disable_file_input && $orig_img_url): ?>
        <div class="img-preview-label">Preview</div>
        <img src="<?= htmlspecialchars($orig_img_url) ?>" class="img-fluid img-preview">
        <div class="mt-2">
          <button type="button" class="btn btn-link p-0" onclick="enableFileInput()">Choose another file</button>
        </div>
      <?php endif; ?>
    </div>
  </div>
  <div class="mb-3">
    <select class="form-control" id="compression_level" name="compression_level" required>
      <option value="auto"<?= ($selected_compression_level=='auto'?' selected':'') ?>>Auto (Recommended)</option>
      <option value="high"<?= ($selected_compression_level=='high'?' selected':'') ?>>High Compression</option>
      <option value="medium"<?= ($selected_compression_level=='medium'?' selected':'') ?>>Medium Compression</option>
      <option value="low"<?= ($selected_compression_level=='low'?' selected':'') ?>>Low Compression</option>
    </select>
    <small class="form-text text-muted">Select compression level</small>
  </div>
  <div class="row">
    <div class="col-12 d-flex justify-content-center">
      <button class="btn" type="submit" id="compress-btn" style="margin-top:10px;width:100%;max-width:320px;">Compress SVG</button>
    </div>
  </div>
</form>
         <div id="compressor-results-area" class="mt-5 mb-4 d-flex justify-content-center" aria-live="polite">
           <?= $compression_result ?>
         </div>
        </div>
      </div>
    </div>
  </section>

  <!-- HOW IT WORKS SECTION -->
  <section class="wrapper">
    <div class="container pt-2 pt-md-7 pb-7 pb-md-8 text-center">
      <h2 class="display-4 mb-3">How SVG Image Compression Works</h2>
      <div class="row gx-lg-8 gx-xl-12 gy-6 process-wrapper line mt-5">
        <div class="col-md-6 col-lg-4">
          <span class="icon btn btn-circle btn-lg btn-soft-primary pe-none mb-4"><span class="number">01</span></span>
          <h4 class="mb-1">Upload Your SVG File</h4>
          <p class="mb-0">
            Start by choosing any SVG file from your device. Our tool instantly checks your file for compatibility and prepares it for optimal compression. No technical skills required—just drag, drop, and go.
          </p>
        </div>
        <div class="col-md-6 col-lg-4">
          <span class="icon btn btn-circle btn-lg btn-primary pe-none mb-4"><span class="number">02</span></span>
          <h4 class="mb-1">Select Compression Level</h4>
          <p class="mb-0">
            Pick your preferred compression level: Auto, High, Medium, or Low. "Auto" uses smart algorithms to balance size and quality, while other options give you full control over the final output. You can re-compress with different settings any time.
          </p>
        </div>
        <div class="col-md-6 col-lg-4">
          <span class="icon btn btn-circle btn-lg btn-soft-primary pe-none mb-4"><span class="number">03</span></span>
          <h4 class="mb-1">Download & Use Instantly</h4>
          <p class="mb-0">
            In seconds, your optimized SVG file is ready for download—smaller, faster, and still sharp. Use it for websites, apps, presentations, or print without sacrificing quality or code clarity.
          </p>
        </div>
      </div>
      <div class="row mt-5">
        <div class="col-lg-10 mx-auto">
          <div class="alert alert-primary p-4 fs-17">
            <strong>Tip:</strong> Compressing SVG files before uploading to your website or sharing online can dramatically improve load times, SEO, and user experience while saving storage and bandwidth.
          </div>
        </div>
      </div>
    </div>
  </section>

  <!-- FEATURE CHECKLIST -->
  <section class="wrapper">
    <div class="container pt-2 pb-5 text-center">
      <p class="lead fs-18 lh-sm mb-4">
        Whether you’re a web developer, designer, illustrator, or everyday user, our SVG Compressor makes it easy to optimize your vector graphics with just a few clicks. No technical skills required—just fast, reliable results every time.
      </p>
      <div class="row justify-content-center">
        <div class="col-md-6">
          <ul class="feature-checklist">
            <li>Supports all SVG files—icons, logos, illustrations, and graphics</li>
            <li>No watermarks, no sign-up, unlimited free use</li>
            <li>Smart compression: balances file size and visual fidelity automatically</li>
            <li>Works on mobile, tablet, and desktop devices</li>
            <li>Secure: your SVG files are never stored or shared</li>
          </ul>
        </div>
        <div class="col-md-6">
          <ul class="feature-checklist">
            <li>Instant results—see file size reduction in real time</li>
            <li>Drag & drop or browse to upload SVGs</li>
            <li>Batch compression for multiple files at once</li>
            <li>Optimized for web, email, print, and presentations</li>
            <li>Trusted by designers, developers, and creators worldwide</li>
          </ul>
        </div>
      </div>
      <div class="alert alert-success p-4 fs-17 mt-4">
        <strong>Ready to optimize your SVG files?</strong> Compress your first vector image now—no registration required!
      </div>
    </div>
  </section>

  <!-- WHY SVG COMPRESSION MATTERS -->
  <section class="wrapper bg-gradient-primary">
    <div class="container pt-8 pt-md-5 pb-5 pb-md-2">
      <h2 class="display-4 mb-3">Why Use an SVG Compressor?</h2>
      <p class="lead fs-18 lh-sm mb-7">
        SVG (Scalable Vector Graphics) is the universal format for high-quality, resolution-independent graphics on the web. SVGs are everywhere—icons, logos, illustrations, infographics, and UI elements. But even SVG files can become bloated with unnecessary code, metadata, and whitespace, slowing down your website and increasing bandwidth costs. Our SVG Compressor strips away the excess, giving you lean, fast-loading vector graphics that look perfect on any device or screen size.
      </p>
      <div class="row gx-lg-8 gx-xl-12 gy-6">
        <div class="col-md-6 col-lg-4">
          <div class="card shadow-sm p-4 h-100">
            <h5 class="mb-2">Boost Website Speed</h5>
            <p class="mb-0">
              Compressed SVG files load instantly, making your website feel snappy and responsive. Google ranks faster sites higher, so SVG optimization is a must for every site owner and developer.
            </p>
          </div>
        </div>
        <div class="col-md-6 col-lg-4">
          <div class="card shadow-sm p-4 h-100">
            <h5 class="mb-2">Save Storage & Bandwidth</h5>
            <p class="mb-0">
              Smaller SVGs mean less server space, lower hosting costs, and reduced data usage for your visitors. Whether you run a blog, store, or SaaS platform, SVG compression pays off.
            </p>
          </div>
        </div>
        <div class="col-md-6 col-lg-4">
          <div class="card shadow-sm p-4 h-100">
            <h5 class="mb-2">Cleaner Code & Security</h5>
            <p class="mb-0">
              Our compressor removes hidden code, metadata, and comments, reducing the risk of malicious scripts and making your SVGs easier to maintain, edit, and animate.
            </p>
          </div>
        </div>
      </div>
      <div class="row mt-6">
        <div class="col-lg-10 mx-auto">
          <div class="alert alert-primary p-4 fs-17">
            <strong>Pro Tip:</strong> Always compress SVGs before uploading to your website, app, or design system. This one step can dramatically improve performance and security.
          </div>
        </div>
      </div>
    </div>
  </section>

  <!-- HOW SVG COMPRESSION WORKS (IN-DEPTH) -->
  <section class="wrapper bg-light">
    <div class="container pt-8 pt-md-10 pb-8 pb-md-12">
      <h2 class="display-4 mb-3">How SVG Compression Works</h2>
      <div class="row gx-lg-8 gx-xl-12 gy-6">
        <div class="col-md-6 text-start">
          <p class="fs-18 lh-lg">
            SVG files are written in XML—a text-based markup language. Over time, SVGs can accumulate unnecessary whitespace, comments, metadata, and unused elements, especially when exported from design tools like Adobe Illustrator, Figma, or Sketch. These extra bytes slow down your site and can even pose security risks.
          </p>
          <p class="fs-18 lh-lg">
            Our SVG Compressor uses advanced algorithms and the industry-standard SVGO tool to remove all redundant code, streamline paths, and optimize your SVGs for the web. The result? Smaller files, faster load times, and SVGs that are easier to animate, style, and maintain.
          </p>
          <p class="fs-18 lh-lg">
            Choose your compression level: "Auto" for smart optimization, or High, Medium, and Low for more control. "High" strips out everything but the essentials, while "Low" preserves more structure for editing. You can preview your SVG before and after compression to ensure perfect results.
          </p>
        </div>
        <div class="col-md-6 text-start">
          <ul class="icon-list bullet-bg bullet-soft-green fs-18 lh-lg">
            <li><strong>Whitespace Removal:</strong> Deletes all unnecessary spaces and line breaks for a leaner file.</li>
            <li><strong>Comment & Metadata Stripping:</strong> Removes hidden comments, IDs, and metadata that bloat SVGs.</li>
            <li><strong>Path Simplification:</strong> Streamlines vector paths and shapes for smaller, faster SVGs.</li>
            <li><strong>Security:</strong> Cleans up potentially unsafe code and scripts embedded in SVGs.</li>
            <li><strong>Smart Optimization:</strong> Balances file size and code readability for easy editing and animation.</li>
          </ul>
        </div>
      </div>
      <div class="row mt-5">
        <div class="col-lg-10 mx-auto">
          <div class="alert alert-success p-4 fs-17">
            <strong>Did you know?</strong> SVGs compressed with our tool can be up to 80% smaller than unoptimized exports from design software. That means lightning-fast sites and apps!
          </div>
        </div>
      </div>
    </div>
  </section>

  <!-- CUSTOMER REVIEWS SECTION -->
  <section class="wrapper" style="background-color:#f5faff;color:#23272b;">
    <div class="container py-7">
      <h2 class="display-6 mb-5 text-center" style="color:#212121;">What Users Say About Our SVG Compressor</h2>
      <div class="row gx-5 gy-4 justify-content-center">
        <div class="col-md-4">
          <div class="card shadow-sm p-4 h-100" style="background-color:#fff;color:#23272b;border:1px solid #e6e6e6;">
            <div class="mb-2 text-warning">
              <i class="bi bi-star-fill"></i><i class="bi bi-star-fill"></i><i class="bi bi-star-fill"></i><i class="bi bi-star-fill"></i><i class="bi bi-star-fill"></i>
            </div>
            <p>
              “The SVG Compressor has made my design workflow so much smoother. I can compress dozens of icons and illustrations in seconds, and the results are always perfect. Highly recommended for any designer!”
            </p>
            <div class="fw-bold" style="color:#2586f7;">– Emma, UI/UX Designer</div>
          </div>
        </div>
        <div class="col-md-4">
          <div class="card shadow-sm p-4 h-100" style="background-color:#fff;color:#23272b;border:1px solid #e6e6e6;">
            <div class="mb-2 text-warning">
              <i class="bi bi-star-fill"></i><i class="bi bi-star-fill"></i><i class="bi bi-star-fill"></i><i class="bi bi-star-fill"></i><i class="bi bi-star-half"></i>
            </div>
            <p>
              “I use SVGs for all my web projects. This compressor helps me keep load times low and code clean. The batch feature is a huge time saver, and my site scores have never been better.”
            </p>
            <div class="fw-bold" style="color:#2586f7;">– Raj, Frontend Developer</div>
          </div>
        </div>
        <div class="col-md-4">
          <div class="card shadow-sm p-4 h-100" style="background-color:#fff;color:#23272b;border:1px solid #e6e6e6;">
            <div class="mb-2 text-warning">
              <i class="bi bi-star-fill"></i><i class="bi bi-star-fill"></i><i class="bi bi-star-fill"></i><i class="bi bi-star-fill"></i><i class="bi bi-star-fill"></i>
            </div>
            <p>
              “As a marketer, I love how this tool keeps my infographics crisp and lightweight. The interface is clean, and I can compress SVGs for email, web, and print in one place.”
            </p>
            <div class="fw-bold" style="color:#2586f7;">– Chloe, Digital Marketer</div>
          </div>
        </div>
      </div>
      <div class="row gx-5 gy-4 justify-content-center mt-4">
        <div class="col-md-4">
          <div class="card shadow-sm p-4 h-100" style="background-color:#fff;color:#23272b;border:1px solid #e6e6e6;">
            <div class="mb-2 text-warning">
              <i class="bi bi-star-fill"></i><i class="bi bi-star-fill"></i><i class="bi bi-star-fill"></i><i class="bi bi-star-fill"></i><i class="bi bi-star-fill"></i>
            </div>
            <p>
              “No more bloated SVG files! I can compress graphics for my presentations and newsletters in seconds. The results are always reliable, and the tool is super easy to use.”
            </p>
            <div class="fw-bold" style="color:#2586f7;">– Lucas, Business Analyst</div>
          </div>
        </div>
        <div class="col-md-4">
          <div class="card shadow-sm p-4 h-100" style="background-color:#fff;color:#23272b;border:1px solid #e6e6e6;">
            <div class="mb-2 text-warning">
              <i class="bi bi-star-fill"></i><i class="bi bi-star-fill"></i><i class="bi bi-star-fill"></i><i class="bi bi-star-fill"></i><i class="bi bi-star-fill"></i>
            </div>
            <p>
              “The drag-and-drop feature is fantastic. I can compress dozens of SVG icons for my app in one go, and the code stays clean and readable. My team loves it.”
            </p>
            <div class="fw-bold" style="color:#2586f7;">– Mia, App Developer</div>
          </div>
        </div>
      </div>
    </div>
  </section>

  <!-- ORGANIC GROWTH TIPS -->
  <section class="wrapper bg-light">
    <div class="container pt-8 pt-md-10 pb-8 pb-md-12">
      <h2 class="display-4 mb-3">Tips for Effective SVG Compression</h2>
      <div class="row gx-lg-8 gx-xl-12 gy-6">
        <div class="col-md-6 text-start">
          <ul class="icon-list bullet-bg bullet-soft-green fs-18 lh-lg">
            <li><strong>Clean Up Before Export:</strong> Remove unused layers, groups, and hidden elements in your design tool before saving as SVG.</li>
            <li><strong>Use Descriptive IDs:</strong> For interactive SVGs, keep IDs meaningful and short to reduce code size and aid editing.</li>
            <li><strong>Minimize Complexity:</strong> Simplify paths, reduce points, and avoid unnecessary gradients or filters for smaller SVGs.</li>
            <li><strong>Preview Before Download:</strong> Always compare the original and compressed SVG to ensure visuals and interactivity are preserved.</li>
            <li><strong>Keep Originals Safe:</strong> Store uncompressed SVGs for future editing, and use compressed versions for web and sharing.</li>
          </ul>
        </div>
        <div class="col-md-6 text-start">
          <ul class="icon-list bullet-bg bullet-soft-green fs-18 lh-lg">
            <li><strong>Optimize for Accessibility:</strong> Add titles, descriptions, and ARIA roles to SVGs for better SEO and usability.</li>
            <li><strong>Test Responsiveness:</strong> Use viewBox and relative units to ensure SVGs scale perfectly on all devices.</li>
            <li><strong>Automate Your Workflow:</strong> Integrate SVG compression into your build process or CMS for consistent optimization.</li>
            <li><strong>Stay Updated:</strong> Follow best practices and new SVG features to ensure your graphics remain modern and efficient.</li>
          </ul>
        </div>
      </div>
      <div class="row mt-5">
        <div class="col-lg-10 mx-auto">
          <div class="alert alert-success p-4 fs-17">
            <strong>Did you know?</strong> Optimized SVGs can cut page load time by up to 80%, leading to better SEO, higher engagement, and lower bounce rates.
          </div>
        </div>
      </div>
    </div>
  </section>

  <!-- FAQ SECTION -->
  <section class="wrapper bg-gradient-primary">
    <div class="container pt-10 pt-md-12">
      <h2 class="display-4 mb-3">Frequently Asked Questions</h2>
      <div class="row gx-lg-8 gx-xl-12 gy-6">
        <div class="col-lg-6 text-start">
          <div class="accordion" id="faqAccordionLeft">
            <div class="accordion-item">
              <h2 class="accordion-header" id="faqHeading1">
                <button class="accordion-button collapsed" type="button" data-bs-toggle="collapse" data-bs-target="#faqCollapse1" aria-expanded="false" aria-controls="faqCollapse1">
                  How does the SVG Compressor work?
                </button>
              </h2>
              <div id="faqCollapse1" class="accordion-collapse collapse" aria-labelledby="faqHeading1" data-bs-parent="#faqAccordionLeft">
                <div class="accordion-body">
                  <p>
                    The SVG Compressor uses advanced algorithms and SVGO to remove unnecessary code, whitespace, comments, and metadata from SVG files. You can choose your preferred compression level, and the tool will balance optimization and code readability, so your graphics look perfect and load instantly.
                  </p>
                </div>
              </div>
            </div>
            <div class="accordion-item">
              <h2 class="accordion-header" id="faqHeading2">
                <button class="accordion-button collapsed" type="button" data-bs-toggle="collapse" data-bs-target="#faqCollapse2" aria-expanded="false" aria-controls="faqCollapse2">
                  Will my SVG files lose quality?
                </button>
              </h2>
              <div id="faqCollapse2" class="accordion-collapse collapse" aria-labelledby="faqHeading2" data-bs-parent="#faqAccordionLeft">
                <div class="accordion-body">
                  <p>
                    No! SVG compression only removes unnecessary code and data. Your graphics will remain crisp, scalable, and visually identical to the original. You can preview the result before downloading to ensure it meets your standards.
                  </p>
                </div>
              </div>
            </div>
            <div class="accordion-item">
              <h2 class="accordion-header" id="faqHeading3">
                <button class="accordion-button collapsed" type="button" data-bs-toggle="collapse" data-bs-target="#faqCollapse3" aria-expanded="false" aria-controls="faqCollapse3">
                  What SVG files are supported?
                </button>
              </h2>
              <div id="faqCollapse3" class="accordion-collapse collapse" aria-labelledby="faqHeading3" data-bs-parent="#faqAccordionLeft">
                <div class="accordion-body">
                  <p>
                    This tool supports all valid SVG files—icons, logos, illustrations, infographics, and more. For other formats like PNG, JPG, GIF, or WebP, please use our dedicated compressors.
                  </p>
                </div>
              </div>
            </div>
            <div class="accordion-item">
              <h2 class="accordion-header" id="faqHeading4">
                <button class="accordion-button collapsed" type="button" data-bs-toggle="collapse" data-bs-target="#faqCollapse4" aria-expanded="false" aria-controls="faqCollapse4">
                  Is there a file size or usage limit?
                </button>
              </h2>
              <div id="faqCollapse4" class="accordion-collapse collapse" aria-labelledby="faqHeading4" data-bs-parent="#faqAccordionLeft">
                <div class="accordion-body">
                  <p>
                    No, you can compress as many SVG files as you like, up to 6MB per file. There are no watermarks, no sign-up, and no hidden restrictions—just fast, free compression.
                  </p>
                </div>
              </div>
            </div>
            <div class="accordion-item">
              <h2 class="accordion-header" id="faqHeading5">
                <button class="accordion-button collapsed" type="button" data-bs-toggle="collapse" data-bs-target="#faqCollapse5" aria-expanded="false" aria-controls="faqCollapse5">
                  Is my privacy protected?
                </button>
              </h2>
              <div id="faqCollapse5" class="accordion-collapse collapse" aria-labelledby="faqHeading5" data-bs-parent="#faqAccordionLeft">
                <div class="accordion-body">
                  <p>
                    Yes, your SVG files are processed securely and are never stored or shared. All compression happens in real-time, and your files are deleted immediately after processing.
                  </p>
                </div>
              </div>
            </div>
          </div>
        </div>
        <!--/col-->
        <div class="col-lg-6 text-start">
          <div class="accordion" id="faqAccordionRight">
            <div class="accordion-item">
              <h2 class="accordion-header" id="faqHeading6">
                <button class="accordion-button collapsed" type="button" data-bs-toggle="collapse" data-bs-target="#faqCollapse6" aria-expanded="false" aria-controls="faqCollapse6">
                  Can I compress multiple SVG files at once?
                </button>
              </h2>
              <div id="faqCollapse6" class="accordion-collapse collapse" aria-labelledby="faqHeading6" data-bs-parent="#faqAccordionRight">
                <div class="accordion-body">
                  <p>
                    Absolutely! Use the batch compression feature to upload and optimize several SVG files in one go. This is perfect for web projects, design systems, or anyone managing lots of graphics.
                  </p>
                </div>
              </div>
            </div>
            <div class="accordion-item">
              <h2 class="accordion-header" id="faqHeading7">
                <button class="accordion-button collapsed" type="button" data-bs-toggle="collapse" data-bs-target="#faqCollapse7" aria-expanded="false" aria-controls="faqCollapse7">
                  Will the tool work on my phone or tablet?
                </button>
              </h2>
              <div id="faqCollapse7" class="accordion-collapse collapse" aria-labelledby="faqHeading7" data-bs-parent="#faqAccordionRight">
                <div class="accordion-body">
                  <p>
                    Yes, our SVG compressor is fully responsive and works on all modern mobile devices and tablets. You can compress and download SVG files directly from your phone, anywhere, anytime.
                  </p>
                </div>
              </div>
            </div>
            <div class="accordion-item">
              <h2 class="accordion-header" id="faqHeading8">
                <button class="accordion-button collapsed" type="button" data-bs-toggle="collapse" data-bs-target="#faqCollapse8" aria-expanded="false" aria-controls="faqCollapse8">
                  Does compression affect SEO?
                </button>
              </h2>
              <div id="faqCollapse8" class="accordion-collapse collapse" aria-labelledby="faqHeading8" data-bs-parent="#faqAccordionRight">
                <div class="accordion-body">
                  <p>
                    Yes—in a good way! Smaller SVG files mean faster page loads, which Google rewards with higher rankings. Optimized SVGs also improve user experience and accessibility.
                  </p>
                </div>
              </div>
            </div>
            <div class="accordion-item">
              <h2 class="accordion-header" id="faqHeading9">
                <button class="accordion-button collapsed" type="button" data-bs-toggle="collapse" data-bs-target="#faqCollapse9" aria-expanded="false" aria-controls="faqCollapse9">
                  Can I use compressed SVG files for print?
                </button>
              </h2>
              <div id="faqCollapse9" class="accordion-collapse collapse" aria-labelledby="faqHeading9" data-bs-parent="#faqAccordionRight">
                <div class="accordion-body">
                  <p>
                    For web and digital use, compressed SVG files are perfect. For high-quality print, keep an original uncompressed version, as compression may remove some metadata used in print workflows.
                  </p>
                </div>
              </div>
            </div>
            <div class="accordion-item">
              <h2 class="accordion-header" id="faqHeading10">
                <button class="accordion-button collapsed" type="button" data-bs-toggle="collapse" data-bs-target="#faqCollapse10" aria-expanded="false" aria-controls="faqCollapse10">
                  Is this tool really free?
                </button>
              </h2>
              <div id="faqCollapse10" class="accordion-collapse collapse" aria-labelledby="faqHeading10" data-bs-parent="#faqAccordionRight">
                <div class="accordion-body">
                  <p>
                    Yes! Our SVG compressor is 100% free to use, with no watermarks, usage limits, or hidden fees. Enjoy unlimited, high-quality SVG compression anytime you need it.
                  </p>
                </div>
              </div>
            </div>
          </div>
        </div>
        <!--/col-->
      </div>
      <!--/.row -->
      <div class="row mt-6">
        <div class="col-lg-10 mx-auto">
          <div class="alert alert-primary p-4 fs-17">
            <strong>Still have questions?</strong> Reach out to our support team anytime. We’re here to help you get the best results from your SVG files!
          </div>
        </div>
      </div>
    </div>
  </section>
</div>
<?php include $_SERVER['DOCUMENT_ROOT'].'/admin/footer.php'; ?>
<script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.3/dist/js/bootstrap.bundle.min.js"></script>
<script>
function updateFilenameAndPreview(input) {
    var fileName = "No file chosen";
    var container = document.getElementById('orig-preview-container');
    if (container) container.innerHTML = '';
    if (input.files && input.files.length > 0) {
        fileName = input.files[0].name;
        // Show preview
        if (container) {
            var reader = new FileReader();
            reader.onload = function(e) {
                container.innerHTML = '<div class="img-preview-label">Preview</div><img src="'+e.target.result+'" class="img-fluid img-preview">';
            }
            reader.readAsDataURL(input.files[0]);
        }
    }
    document.getElementById('file-filename').textContent = fileName;
}
function enableFileInput() {
    var fileInput = document.getElementById('image_file');
    var fileNameDiv = document.getElementById('file-filename');
    var previewDiv = document.getElementById('orig-preview-container');
    fileInput.disabled = false;
    fileInput.value = '';
    fileNameDiv.textContent = 'No file chosen';
    if (previewDiv) previewDiv.innerHTML = '';
}
document.addEventListener('DOMContentLoaded', function() {
    var fileInput = document.getElementById('image_file');
    var fileNameDiv = document.getElementById('file-filename');
    var previewDiv = document.getElementById('orig-preview-container');
    if (fileInput && fileNameDiv) {
        if (!fileInput.disabled) {
            fileInput.value = '';
            fileNameDiv.textContent = 'No file chosen';
            if (previewDiv) previewDiv.innerHTML = '';
        }
        fileInput.onchange = function() {
            updateFilenameAndPreview(this);
        };
        fileNameDiv.onclick = function() {
            if (!fileInput.disabled) fileInput.click();
        };
    }
});
</script>
</body>
</html>
