<?php
session_start();

$compressed_dir = __DIR__ . '/compressed/';
$compressed_url = '/tools/compress-image/compressed/';
if (!is_dir($compressed_dir)) mkdir($compressed_dir, 0777, true);

$compression_result = '';
$compression_level_label = '';
$selected_compression_level = 'auto';
$disable_file_input = false;
$orig_img_url = '';
$orig_size = 0;
$orig_ext = '';
$orig_mime = '';

// Helper: Human-readable compression level
function getCompressionLevelLabel($level) {
    switch ($level) {
        case 'high': return 'High';
        case 'medium': return 'Medium';
        case 'low': return 'Low';
        default: return 'Auto (Recommended)';
    }
}

// Helper: Compress PNG with pngquant if available
function compress_pngquant($input, $output, $min_quality = 60, $max_quality = 90) {
    $which = trim(shell_exec("which pngquant"));
    if (!$which) return false;
    $cmd = "pngquant --quality={$min_quality}-{$max_quality} --output ".escapeshellarg($output)." --force --skip-if-larger ".escapeshellarg($input);
    exec($cmd, $out, $ret);
    return ($ret === 0 && file_exists($output));
}

// Helper: Compress SVG using SVGO or manual optimization
function compress_svg($input, $output) {
    $which = trim(shell_exec("which svgo"));
    if ($which) {
        $cmd = "svgo ".escapeshellarg($input)." --output ".escapeshellarg($output);
        exec($cmd, $out, $ret);
        if ($ret === 0 && file_exists($output)) return true;
    }
    $svg_content = file_get_contents($input);
    if ($svg_content === false) return false;
    $svg_content = preg_replace('/>\s+</', '><', $svg_content);
    $svg_content = preg_replace('/<!--.*?-->/', '', $svg_content);
    $svg_content = preg_replace('/\s+/', ' ', $svg_content);
    $svg_content = trim($svg_content);
    $svg_content = preg_replace('/\s*(xmlns:[^=]*="[^"]*")\s*/', '', $svg_content);
    $svg_content = preg_replace('/\s*(id="[^"]*")\s*/', '', $svg_content);
    $svg_content = preg_replace('/\s*(class="[^"]*")\s*/', '', $svg_content);
    return file_put_contents($output, $svg_content) !== false;
}

// Helper: Compress GIF using gifsicle or GD
function compress_gif($input, $output, $compression_level = 'auto') {
    $which = trim(shell_exec("which gifsicle"));
    if ($which) {
        $optimize_level = ($compression_level === 'high') ? 3 : (($compression_level === 'medium') ? 2 : 1);
        if ($compression_level === 'auto') $optimize_level = 2;
        $cmd = "gifsicle --optimize={$optimize_level} --output ".escapeshellarg($output)." ".escapeshellarg($input);
        exec($cmd, $out, $ret);
        if ($ret === 0 && file_exists($output)) return true;
    }
    $img = imagecreatefromgif($input);
    if ($img === false) return false;
    $success = imagegif($img, $output);
    imagedestroy($img);
    return $success;
}

// --- Download handler: serve file if ?download=filename is set ---
if (isset($_GET['download'])) {
    $file = basename($_GET['download']);
    if (!preg_match('/^imgcmp_[a-zA-Z0-9]+\.(jpg|jpeg|png|webp|gif|svg)$/', $file)) {
        http_response_code(400);
        die('Invalid file name.');
    }
    $filepath = $compressed_dir . $file;
    if (!file_exists($filepath)) {
        http_response_code(404);
        die('File not found.');
    }
    $mime_map = [
        'jpg'=>'image/jpeg','jpeg'=>'image/jpeg','png'=>'image/png',
        'webp'=>'image/webp','gif'=>'image/gif','svg'=>'image/svg+xml'
    ];
    $ext = strtolower(pathinfo($file, PATHINFO_EXTENSION));
    $mime = $mime_map[$ext] ?? 'application/octet-stream';
    header('Content-Description: File Transfer');
    header('Content-Type: '.$mime);
    header('Content-Disposition: attachment; filename="' . $file . '"');
    header('Expires: 0');
    header('Cache-Control: must-revalidate');
    header('Pragma: public');
    header('Content-Length: ' . filesize($filepath));
    flush();
    readfile($filepath);
    exit;
}

// --- HANDLE POST: Compress and Redirect ---
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $max_size = 6 * 1024 * 1024; // 6MB
    $compression_level = $_POST['compression_level'] ?? 'auto';
    $compression_level_label = getCompressionLevelLabel($compression_level);
    $selected_compression_level = $compression_level;
    $success = false;
    $unique_id = uniqid('imgcmp_');
    $download_name = '';
    $compressed_file_path = '';
    $compressed_img_url = '';
    $orig_size = 0;

    // If new file uploaded, handle upload and store path in session
    if (isset($_FILES['image_file']) && $_FILES['image_file']['error'] === UPLOAD_ERR_OK) {
        $file = $_FILES['image_file'];
        if ($file['size'] > $max_size) {
            $_SESSION['compression_result'] = '<div class="alert alert-danger">File too large. Maximum size is 6MB.</div>';
        } else {
            $img_info = getimagesize($file['tmp_name']);
            $mime = $img_info['mime'];
            if (in_array($mime, ['image/jpeg','image/jpg','image/png','image/webp','image/gif','image/svg+xml'])) {
                $ext_map = [
                    'image/jpeg'=>'.jpg', 'image/jpg'=>'.jpg', 'image/png'=>'.png',
                    'image/webp'=>'.webp', 'image/gif'=>'.gif', 'image/svg+xml'=>'.svg'
                ];
                $orig_ext = $ext_map[$mime];
                $orig_name = $unique_id . '_orig' . $orig_ext;
                $orig_path = $compressed_dir . $orig_name;
                move_uploaded_file($file['tmp_name'], $orig_path);
                $_SESSION['uploaded_img_path'] = $orig_path;
                $_SESSION['uploaded_img_url']  = $compressed_url . $orig_name;
                $_SESSION['uploaded_img_size'] = filesize($orig_path);
                $_SESSION['uploaded_img_mime'] = $mime;
                $orig_img_url = $_SESSION['uploaded_img_url'];
                $orig_size = $_SESSION['uploaded_img_size'];
                $orig_mime = $mime;
                $disable_file_input = true;
            } else {
                $_SESSION['compression_result'] = '<div class="alert alert-danger">Only JPG, PNG, WebP, GIF, and SVG images are supported.</div>';
            }
        }
    }
    // If user changed compression level after upload, reuse previous file
    elseif (isset($_SESSION['uploaded_img_path']) && file_exists($_SESSION['uploaded_img_path'])) {
        $orig_path = $_SESSION['uploaded_img_path'];
        $orig_img_url = $_SESSION['uploaded_img_url'];
        $orig_size = $_SESSION['uploaded_img_size'];
        $orig_mime = $_SESSION['uploaded_img_mime'];
        $orig_ext = strtolower(pathinfo($orig_path, PATHINFO_EXTENSION));
        $disable_file_input = true;
    }

    // If we have a valid original file, proceed to compress
    if (!empty($orig_path) && file_exists($orig_path)) {
        $download_name = $unique_id . $orig_ext;
        $compressed_file_path = $compressed_dir . $download_name;

        if ($orig_mime === 'image/jpeg' || $orig_mime === 'image/jpg') {
            $img = imagecreatefromjpeg($orig_path);
            $quality = ($compression_level === 'high') ? 40 : (($compression_level === 'medium') ? 60 : 80);
            if ($compression_level === 'auto') $quality = 60;
            $success = imagejpeg($img, $compressed_file_path, $quality);
            if (isset($img) && is_resource($img)) imagedestroy($img);
        } elseif ($orig_mime === 'image/png') {
            $quality = ($compression_level === 'high') ? 60 : (($compression_level === 'medium') ? 75 : 90);
            if ($compression_level === 'auto') $quality = 80;
            // Try pngquant first
            $pngquant_success = compress_pngquant($orig_path, $compressed_file_path, $quality, 100);
            if ($pngquant_success) {
                $success = true;
            } else {
                $img = imagecreatefrompng($orig_path);
                $gd_quality = ($compression_level === 'high') ? 9 : (($compression_level === 'medium') ? 6 : 3);
                if ($compression_level === 'auto') $gd_quality = 6;
                $success = imagepng($img, $compressed_file_path, $gd_quality);
                if (isset($img) && is_resource($img)) imagedestroy($img);
            }
        } elseif ($orig_mime === 'image/webp') {
            $img = imagecreatefromwebp($orig_path);
            $quality = ($compression_level === 'high') ? 40 : (($compression_level === 'medium') ? 60 : 80);
            if ($compression_level === 'auto') $quality = 60;
            $success = imagewebp($img, $compressed_file_path, $quality);
            if (isset($img) && is_resource($img)) imagedestroy($img);
        } elseif ($orig_mime === 'image/gif') {
            $success = compress_gif($orig_path, $compressed_file_path, $compression_level);
        } elseif ($orig_mime === 'image/svg+xml') {
            $success = compress_svg($orig_path, $compressed_file_path);
        }
    }

    // Handle results
    if ($success && file_exists($compressed_file_path)) {
        $compressed_size = filesize($compressed_file_path);
        $compressed_img_url = $compressed_url . $download_name;
        if ($compressed_size >= $orig_size) {
            $_SESSION['compression_result'] = '
            <div class="result-card main-result-card">
              <div class="result-meta">Compression Results</div>
              <div class="row gx-3 gy-3 align-items-center justify-content-center mb-3 flex-wrap result-img-row">
                <div class="col-12 col-md-6 text-center mb-2 mb-md-0">
                  <div class="img-preview-label">Original</div>
                  <img src="'.htmlspecialchars($orig_img_url).'" class="img-fluid img-preview" alt="Original Image" style="max-width:180px;max-height:180px;border-radius:10px;border:1px solid #eee;">
                </div>
                <div class="col-12 col-md-6 text-center">
                  <div class="img-preview-label">Original (Optimized)</div>
                  <img src="'.htmlspecialchars($orig_img_url).'" class="img-fluid img-preview" alt="Original Image" style="max-width:180px;max-height:180px;border-radius:10px;border:1px solid #eee;">
                </div>
              </div>
              <div class="result-details">
                <div class="result-row flex-column flex-md-row text-center text-md-start">
                  <span class="result-label">Compression Level</span>
                  <span class="result-value">'.htmlspecialchars($compression_level_label).'</span>
                </div>
                <div class="result-row flex-column flex-md-row text-center text-md-start">
                  <span class="result-label">Original Size</span>
                  <span class="result-value">'.number_format($orig_size/1024, 2).' KB</span>
                </div>
                <div class="result-row flex-column flex-md-row text-center text-md-start">
                  <span class="result-label">Final Size</span>
                  <span class="result-value">'.number_format($orig_size/1024, 2).' KB</span>
                </div>
                <div class="result-row flex-column flex-md-row text-center text-md-start">
                  <span class="result-label">Reduction</span>
                  <span class="result-value">0%</span>
                </div>
              </div>
              <div class="alert alert-info mt-3 mb-0">
                <strong>No further compression possible.</strong> The image is already optimized.
              </div>
              <a href="?download='.urlencode(basename($orig_path)).'" class="btn btn-primary mt-3">Download Original</a>
            </div>';
        } else {
            $percent = $orig_size > 0 ? round(100 - ($compressed_size / $orig_size * 100), 1) : 0;
            $_SESSION['compression_result'] = '
            <div class="result-card main-result-card">
              <div class="result-meta">Compression Results</div>
              <div class="row gx-3 gy-3 align-items-center justify-content-center mb-3 flex-wrap result-img-row">
                <div class="col-12 col-md-6 text-center mb-2 mb-md-0">
                  <div class="img-preview-label">Original</div>
                  <img src="'.htmlspecialchars($orig_img_url).'" class="img-fluid img-preview" alt="Original Image" style="max-width:180px;max-height:180px;border-radius:10px;border:1px solid #eee;">
                </div>
                <div class="col-12 col-md-6 text-center">
                  <div class="img-preview-label">Compressed</div>
                  <img src="'.htmlspecialchars($compressed_img_url).'" class="img-fluid img-preview" alt="Compressed Image" style="max-width:180px;max-height:180px;border-radius:10px;border:1px solid #eee;">
                </div>
              </div>
              <div class="result-details">
                <div class="result-row flex-column flex-md-row text-center text-md-start">
                  <span class="result-label">Compression Level</span>
                  <span class="result-value">'.htmlspecialchars($compression_level_label).'</span>
                </div>
                <div class="result-row flex-column flex-md-row text-center text-md-start">
                  <span class="result-label">Original Size</span>
                  <span class="result-value">'.number_format($orig_size/1024, 2).' KB</span>
                </div>
                <div class="result-row flex-column flex-md-row text-center text-md-start">
                  <span class="result-label">Compressed Size</span>
                  <span class="result-value">'.number_format($compressed_size/1024, 2).' KB</span>
                </div>
                <div class="result-row flex-column flex-md-row text-center text-md-start">
                  <span class="result-label">Reduction</span>
                  <span class="result-value">'.$percent.'%</span>
                </div>
              </div>
              <a href="?download='.urlencode($download_name).'" class="btn btn-primary mt-3">Download Compressed</a>
            </div>';
        }
    } elseif (!isset($_SESSION['compression_result'])) {
        $_SESSION['compression_result'] = '<div class="alert alert-danger">Compression failed. Please try another image.</div>';
    }
    header("Location: ".$_SERVER['REQUEST_URI']);
    exit;
}

// On GET, show results only if they exist in session, then clear them
if (isset($_SESSION['compression_result'])) {
    $compression_result = $_SESSION['compression_result'];
    unset($_SESSION['compression_result']);
    if (isset($_SESSION['uploaded_img_url'])) {
        $orig_img_url = $_SESSION['uploaded_img_url'];
        $disable_file_input = true;
    }
    if (isset($_SESSION['compression_level'])) {
        $selected_compression_level = $_SESSION['compression_level'];
    }
} else {
    $compression_result = '';
    $disable_file_input = false;
    $orig_img_url = '';
}


// Now include your HTML and header/footer files
$page_title = "Image Compressor: Instantly Optimize Your Images Online";
$page_description = "Compress and optimize your images instantly with our free Image Compressor. Reduce file size, maintain quality, and boost website speed. No signup, no watermark, unlimited use.";
$page_keywords = "Image Compressor, online image compression, compress images, reduce image size, JPEG compressor, PNG compressor, image optimizer, photo compressor, web image optimization, free image compressor";

include $_SERVER['DOCUMENT_ROOT'].'/admin/header.php';
?>
<!-- Your HTML and form go here -->


<html>
<head>
    <link rel="stylesheet" href="/assets/css/plugins.css" />
    <link rel="stylesheet" href="/assets/css/base.css" />
    <link rel="stylesheet" href="/assets/css/style.css" />
    <link rel="preload" href="/assets/css/fonts/space.css" as="style" onload="this.rel='stylesheet'" />
<style>
/* ================================
   Loader Overlay (Modal Spinner)
   ================================ */
#modal-loader-overlay {
    display: none;
    position: fixed;
    z-index: 1050;
    top: 0; left: 0; right: 0; bottom: 0;
    background: rgba(255,255,255,0.92);
    align-items: center;
    justify-content: center;
}
#modal-loader-overlay .loader-content {
    display: flex;
    align-items: center;
    flex-direction: column;
}
#modal-loader-overlay .spinner-border {
    width: 3rem;
    height: 3rem;
}
#modal-loader-overlay .loader-text {
    margin-top: 1rem;
    font-size: 1.2rem;
    color: #2586f7;
    font-weight: 500;
}

/* ================================
   Form Inputs & Button Styling
   ================================ */
form#image-compressor-form input.form-control,
form#image-compressor-form select.form-control {
    height: 56px;
    font-size: 18px;
    padding: 12px 16px;
    display: block;
    width: 100%;
    font-weight: 700;
    line-height: 2.0;
    color: #60697b;
    background-color: var(--bs-body-bg, #fff);
    border: 1px solid rgba(8, 60, 130, 0.07);
    border-radius: 0.4rem;
    box-shadow: 0 0 1.25rem rgba(30, 34, 40, 0.04);
    transition: border-color 0.15s, box-shadow 0.15s;
}
form#image-compressor-form button#compress-btn {
    display: flex;                   /* Enable flexbox for centering */
    align-items: center;             /* Vertically center content */
    justify-content: center;         /* Horizontally center content */
    height: 54px;
    font-size: 1.15rem;
    line-height: 1.2;
    background: linear-gradient(90deg, #3f78e0 0%, #6a82fb 100%);
    color: #fff;
    font-weight: 600;
    border-radius: 6px;
    width: 100%;
    max-width: 320px;
    margin: 14px auto 0;
    border: none;
    box-shadow: 0 0 1.25rem rgba(30, 34, 40, 0.04);
    cursor: pointer;
    /* Remove these if present: */
    padding-top: 0;
    padding-bottom: 0;
    padding-left: 0;
    padding-right: 0;
    padding: 0 24px;

}

/* ================================
   Custom File Input Styling
   ================================ */
.file-input-row,
.custom-file-input-wrapper {
    display: flex;
    align-items: center;
    width: 100%;
    max-width: 420px;
    margin: 0 auto 1.5rem auto;
    height: 56px;
    position: relative;
}
.custom-file-input {
    display: none;
}
.custom-file-label {
    background: linear-gradient(90deg, #3f78e0 0%, #6a82fb 100%);
    color: #fff;
    font-weight: 600;
    border-radius: 6px 0 0 6px;
    height: 56px;
    width: 140px;
    font-size: 1.1rem;
    cursor: pointer;
    display: flex;
    align-items: center;
    justify-content: center;
    border: none;
    margin: 0;
    text-align: center;
    transition: background 0.2s;
}
.custom-file-label:hover {
    background: linear-gradient(90deg, #6a82fb 0%, #3f78e0 100%);
}
.custom-file-filename {
    background: #f8fafc;
    color: #60697b;
    border-radius: 0 6px 6px 0;
    height: 56px;
    flex: 1 1 0%;
    font-size: 1.08rem;
    display: flex;
    align-items: center;
    padding-left: 18px;
    font-weight: 500;
    border-left: 1px solid #e0eafc;
    overflow: hidden;
    text-overflow: ellipsis;
    white-space: nowrap;
}

/* ================================
   Custom Select Styling (Matches File Input)
   ================================ */
.custom-select-wrapper {
    width: 100%;
    max-width: 420px;
    margin: 0 auto;
    height: 56px;
}
.custom-select-wrapper select.form-control {
    width: 100%;
    height: 56px;
    font-size: 1.1rem;
    font-weight: 600;
    border-radius: 6px;
    background: #f8fafc;
    color: #60697b;
    border: 1px solid #e0eafc;
    box-shadow: 0 0 1.25rem rgba(30, 34, 40, 0.04);
    padding-left: 16px;
    padding-right: 16px;
}

/* ================================
   Helper Texts (Under Inputs)
   ================================ */
.custom-form-text,
#image-compressor-form .form-text {
    font-size: 1.08rem !important;
    font-weight: 600;
    color: #3f78e0 !important;
    margin-bottom: 0;
}

/* ================================
   Image Preview Styling
   ================================ */
.img-preview-label {
    font-size: 0.95rem;
    font-weight: 600;
    margin-bottom: 0.5rem;
    color: #3f78e0;
}
.img-preview {
    max-width: 180px;
    max-height: 180px;
    border-radius: 10px;
    border: 1px solid #eee;
}

/* ================================
   Result Card Styling
   ================================ */
.result-card {
    background: #fff;
    border-radius: 18px;
    box-shadow: 0 8px 32px rgba(60,90,130,0.10), 0 1.5px 4px rgba(60,90,130,0.10);
    max-width: 420px;
    margin: 2rem auto 0 auto;
    padding: 2rem 2rem 1.5rem 2rem;
    display: flex;
    flex-direction: column;
    align-items: center;
    transition: box-shadow 0.2s;
}
.result-card:hover {
    box-shadow: 0 16px 48px rgba(60,90,130,0.13), 0 3px 8px rgba(60,90,130,0.13);
}
.result-meta {
    font-weight: bold;
    font-size: 1.3rem;
    color: #2a3a5c;
    margin-bottom: 0.75rem;
    padding-bottom: 0.25rem;
    border-bottom: 2px solid #3f78e0;
    letter-spacing: 0.02em;
    text-align: center;
    background: linear-gradient(90deg, #f6fafd 70%, #e6f0ff 100%);
    border-radius: 6px 6px 0 0;
}
.result-details {
    width: 100%;
    display: flex;
    flex-direction: column;
    gap: 0.7rem;
}
.result-row {
    display: flex;
    justify-content: space-between;
    align-items: center;
    background: #f6fafd;
    border-radius: 9px;
    padding: 1rem 1.2rem;
    box-shadow: 0 0.5px 1.5px rgba(60,90,130,0.04);
    font-size: 1.11rem;
    max-width: 100%;
    word-wrap: break-word;
    overflow-wrap: break-word;
    white-space: normal;
    flex-direction: column;
    align-items: flex-start;
    text-align: left;
}
.result-label {
    font-weight: 700;
    color: #2a3a5c;
    order: 2;
    margin-top: 0.1em;
    font-size: 1em;
}
.result-value {
    font-weight: 500;
    color: #384c6c;
    text-align: right;
    word-break: break-word;
    order: 1;
    font-size: 1.2em;
    font-weight: 700;
    margin-bottom: 0.15em;
}
.result-earnings {
    font-size: 2.2rem;
    font-weight: 800;
    color: #19b34a;
    margin-bottom: 0.25rem;
    text-align: center;
    letter-spacing: -1px;
    word-break: break-all;
    overflow-wrap: break-word;
    white-space: normal;
    max-width: 100%;
}
.main-result-card {
    max-width: 600px;
    width: 100%;
}
.result-img-row {
    margin-bottom: 1.5rem !important;
}

/* ================================
   Responsive Adjustments
   ================================ */
@media (max-width: 600px) {
    .file-input-row,
    .custom-file-input-wrapper,
    .custom-select-wrapper,
    .result-card,
    .main-result-card {
        max-width: 98vw;
        width: 100%;
    }
    .custom-file-label, .custom-file-filename, .custom-select-wrapper select.form-control {
        font-size: 1rem;
    }
    .custom-file-label {
        width: 38vw;
        min-width: 90px;
        max-width: 50vw;
    }
    .custom-file-filename {
        padding-left: 10px;
    }
    .result-card {
        padding: 1.1rem 0.5rem 1.1rem 0.5rem;
        border-radius: 13px;
        margin: 1.2rem auto 0 auto;
    }
    .result-details {
        gap: 0.5rem;
    }
    .result-row {
        font-size: 1.08rem;
        padding: 0.85rem 0.7rem;
    }
    .result-earnings {
        font-size: 2.1rem;
    }
    .result-meta {
        font-size: 1rem;
    }
    .result-img-row .col-12 {
        margin-bottom: 1.2rem;
    }
}

/* ================================
   Feature Checklist Styling
   ================================ */
.feature-checklist {
    list-style: none;
    padding-left: 0;
    margin-bottom: 1.5rem;
}
.feature-checklist li {
    position: relative;
    padding-left: 2em;
    margin-bottom: 1.1em;
    font-size: 1.08rem;
    font-weight: 500;
    color: #23305c;
}
.feature-checklist li::before {
    content: "\2713";
    position: absolute;
    left: 0;
    top: 0.1em;
    color: #19b34a;
    font-size: 1.25em;
    font-weight: bold;
    line-height: 1;
}


.result-card .result-row {
  flex-wrap: wrap;
  white-space: normal !important;
  word-break: break-word;
}

.result-card .result-value {
  display: block;
  font-weight: bold;
  margin-top: 5px;
  white-space: normal !important;
  word-break: break-word;
}

@media (max-width: 768px) {
  .result-card .result-meta,
  .result-card .result-value,
  .result-card .result-label {
    font-size: 1rem !important;
    white-space: normal !important;
    word-break: break-word;
  }
}

@media (max-width: 767px) {
  .result-card .result-value {
    text-align: left !important;
    display: block;
    white-space: normal;
    word-break: break-word;
    margin-left: 0 !important;
  }
}

.custom-file-input { display: none; }
      .custom-file-label {
        background: linear-gradient(90deg, #3f78e0 0%, #6a82fb 100%);
        color: #fff;
        font-weight: 600;
        border-radius: 6px 0 0 6px;
        height: 56px;
        width: 140px;
        font-size: 1.1rem;
        cursor: pointer;
        display: flex;
        align-items: center;
        justify-content: center;
        border: none;
        margin: 0;
        text-align: center;
        transition: background 0.2s;
      }
      .custom-file-filename {
        background: #f8fafc;
        color: #60697b;
        border-radius: 0 6px 6px 0;
        height: 56px;
        flex: 1 1 0%;
        font-size: 1.08rem;
        display: flex;
        align-items: center;
        padding-left: 18px;
        font-weight: 500;
        border-left: 1px solid #e0eafc;
        overflow: hidden;
        text-overflow: ellipsis;
        white-space: nowrap;
        cursor: pointer;
      }
      .img-preview-label { font-size: 0.95rem; font-weight: 600; margin-bottom: 0.5rem; color: #3f78e0; }
      .img-preview { max-width: 180px; max-height: 180px; border-radius: 10px; border: 1px solid #eee; }
</style>


</head>
<body>
<div class="content-wrapper">

  <!-- Loader Overlay -->
  <div id="modal-loader-overlay">
    <div class="loader-content">
      <div class="spinner-border text-primary" role="status">
        <span class="visually-hidden">Loading...</span>
      </div>
      <span class="loader-text">Compressing Image...</span>
    </div>
  </div>

  <!-- HERO + COMPRESSOR FORM -->
  <section class="wrapper">
    <div class="container pt-2 pt-md-7 pb-7 pb-md-8 text-center">
      <div class="row">
        <div class="col-lg-9 col-xl-8 col-xxl-7 mx-auto" data-group="page-title">
         <h1 class="display-1 ls-sm fs-44 mb-4 px-md-8 px-lg-0">
           Image Compressor <span class="underline-3 style-1 primary"></span>
         </h1>
         <h3 class="lead fs-20 lh-sm mb-4 mt-5">
           Instantly compress your images online for free. Reduce file size, boost website speed, and maintain high quality in just a few clicks—no signup, no watermark, unlimited use.
         </h3>
<form id="image-compressor-form" method="POST" enctype="multipart/form-data" action="" style="max-width:600px;margin:0 auto;">
  <div class="mb-3">
    <div class="custom-file-input-wrapper file-input-row">
      <input
        type="file"
        class="custom-file-input"
        id="image_file"
        name="image_file"
        accept="image/jpeg,image/png,image/webp,image/gif,image/svg+xml"
        <?= $disable_file_input ? 'disabled' : '' ?>
        required
        onchange="updateFilenameAndPreview(this)"
      >
      <label for="image_file" class="custom-file-label" id="choose-file-label">Choose file</label>
      <div class="custom-file-filename" id="file-filename" onclick="if(!document.getElementById('image_file').disabled){document.getElementById('image_file').click();}">No file chosen</div>
    </div>
    <small class="form-text text-muted">Choose an image (JPG, PNG, WebP, GIF, SVG max 6MB)</small>
    <div id="orig-preview-container" class="mt-3">
      <?php if ($disable_file_input && $orig_img_url): ?>
        <div class="img-preview-label">Preview</div>
        <img src="<?= htmlspecialchars($orig_img_url) ?>" class="img-fluid img-preview">
        <div class="mt-2">
          <button type="button" class="btn btn-link p-0" onclick="enableFileInput()">Choose another file</button>
        </div>
      <?php endif; ?>
    </div>
  </div>
  <div class="mb-3">
    <select class="form-control" id="compression_level" name="compression_level" required>
      <option value="auto"<?= ($selected_compression_level=='auto'?' selected':'') ?>>Auto (Recommended)</option>
      <option value="high"<?= ($selected_compression_level=='high'?' selected':'') ?>>High Compression</option>
      <option value="medium"<?= ($selected_compression_level=='medium'?' selected':'') ?>>Medium Compression</option>
      <option value="low"<?= ($selected_compression_level=='low'?' selected':'') ?>>Low Compression</option>
    </select>
    <small class="form-text text-muted">Select compression level</small>
  </div>
  <div class="row">
    <div class="col-12 d-flex justify-content-center">
      <button class="btn" type="submit" id="compress-btn" style="margin-top:10px;width:100%;max-width:320px;">Compress Image</button>
    </div>
  </div>
</form>
         <div id="compressor-results-area" class="mt-5 mb-4 d-flex justify-content-center" aria-live="polite">
           <?= $compression_result ?>
         </div>
        </div>
      </div>
    </div>
  </section>
  <!-- HOW IT WORKS SECTION -->
  <section class="wrapper">
    <div class="container pt-2 pt-md-7 pb-7 pb-md-8 text-center">
      <h2 class="display-4 mb-3">How the Image Compression Works</h2>
      <div class="row gx-lg-8 gx-xl-12 gy-6 process-wrapper line mt-5">
        <div class="col-md-6 col-lg-4">
          <span class="icon btn btn-circle btn-lg btn-soft-primary pe-none mb-4"><span class="number">01</span></span>
          <h4 class="mb-1">Upload Your Image</h4>
          <p class="mb-0">
            Choose any image from your device—JPG, PNG, WebP, GIF, or SVG. Our tool supports all major formats and instantly checks your file for compatibility and optimal compression.
          </p>
        </div>
        <div class="col-md-6 col-lg-4">
          <span class="icon btn btn-circle btn-lg btn-primary pe-none mb-4"><span class="number">02</span></span>
          <h4 class="mb-1">Select Compression Level</h4>
          <p class="mb-0">
            Pick your preferred compression level: Auto, High, Medium, or Low. "Auto" uses smart algorithms to balance size and quality, while other options give you more control over the final output.
          </p>
        </div>
        <div class="col-md-6 col-lg-4">
          <span class="icon btn btn-circle btn-lg btn-soft-primary pe-none mb-4"><span class="number">03</span></span>
          <h4 class="mb-1">Download & Use Instantly</h4>
          <p class="mb-0">
            In seconds, your optimized image is ready for download—smaller, faster, and still sharp. Use it for websites, blogs, social media, or email without sacrificing quality or speed.
          </p>
        </div>
      </div>
      <div class="row mt-5">
        <div class="col-lg-10 mx-auto">
          <div class="alert alert-primary p-4 fs-17">
            <strong>Tip:</strong> Compressing images before uploading to your website or sharing online can dramatically improve load times and user experience, while saving storage and bandwidth.
          </div>
        </div>
      </div>
    </div>
  </section>

  <!-- FEATURE CHECKLIST -->
  <section class="wrapper">
    <div class="container pt-2 pb-5 text-center">
      <p class="lead fs-18 lh-sm mb-4">
        Whether you’re a web developer, designer, blogger, or everyday user, our Image Compressor makes it easy to optimize your photos and graphics with just a few clicks. No technical skills required—just fast, reliable results.
      </p>
      <div class="row justify-content-center">
        <div class="col-md-6">
          <ul class="feature-checklist">
            <li>Supports JPG, PNG, WebP, GIF, SVG, and more</li>
            <li>No watermarks, no sign-up, unlimited free use</li>
            <li>Smart compression: balances quality and file size automatically</li>
            <li>Works on mobile, tablet, and desktop devices</li>
            <li>Secure: your images are never stored or shared</li>
          </ul>
        </div>
        <div class="col-md-6">
          <ul class="feature-checklist">
            <li>Instant results—see file size reduction in real time</li>
            <li>Drag & drop or browse to upload images</li>
            <li>Batch compression for multiple files at once</li>
            <li>Optimized for web, email, and social media sharing</li>
            <li>Trusted by professionals, bloggers, and creators worldwide</li>
          </ul>
        </div>
      </div>
      <div class="alert alert-success p-4 fs-17 mt-4">
        <strong>Ready to optimize your images?</strong> Compress your first photo now—no registration required!
      </div>
    </div>
  </section>

  <!-- WHY IMAGE COMPRESSION MATTERS -->
  <section class="wrapper bg-gradient-primary">
    <div class="container pt-8 pt-md-5 pb-5 pb-md-2">
      <h2 class="display-4 mb-3">Why Compress Images?</h2>
      <p class="lead fs-18 lh-sm mb-7">
        Image compression is crucial for anyone who wants to improve website speed, save storage, and deliver a better user experience. Large, unoptimized images are the number one cause of slow-loading pages, high bounce rates, and wasted bandwidth. By reducing file size without sacrificing quality, you can make your site faster, your emails lighter, and your social posts snappier.
      </p>
      <div class="row gx-lg-8 gx-xl-12 gy-6">
        <div class="col-md-6 col-lg-4">
          <div class="card shadow-sm p-4 h-100">
            <h5 class="mb-2">Boost Website Speed</h5>
            <p class="mb-0">
              Compressed images load faster, which improves SEO, increases conversions, and keeps visitors engaged. Google ranks faster sites higher, so image optimization is a must for every website owner.
            </p>
          </div>
        </div>
        <div class="col-md-6 col-lg-4">
          <div class="card shadow-sm p-4 h-100">
            <h5 class="mb-2">Save Storage & Bandwidth</h5>
            <p class="mb-0">
              Smaller images mean less server space and lower hosting costs. They also reduce data usage for mobile users, making your site more accessible worldwide.
            </p>
          </div>
        </div>
        <div class="col-md-6 col-lg-4">
          <div class="card shadow-sm p-4 h-100">
            <h5 class="mb-2">Enhance User Experience</h5>
            <p class="mb-0">
              Quick-loading images keep users happy, reduce bounce rates, and encourage sharing. Whether you run a blog, store, or portfolio, optimized images are key to a professional, modern web presence.
            </p>
          </div>
        </div>
      </div>
      <div class="row mt-6">
        <div class="col-lg-10 mx-auto">
          <div class="alert alert-primary p-4 fs-17">
            <strong>Pro Tip:</strong> Always compress images before uploading to your website, blog, or email campaigns. This one step can dramatically improve performance and engagement.
          </div>
        </div>
      </div>
    </div>
  </section>

  <!-- CUSTOMER REVIEWS SECTION -->
  <section class="wrapper" style="background-color:#f5faff;color:#23272b;">
    <div class="container py-7">
      <h2 class="display-6 mb-5 text-center" style="color:#212121;">Real Users, Real Results</h2>
      <div class="row gx-5 gy-4 justify-content-center">
        <div class="col-md-4">
          <div class="card shadow-sm p-4 h-100" style="background-color:#fff;color:#23272b;border:1px solid #e6e6e6;">
            <div class="mb-2 text-warning">
              <i class="bi bi-star-fill"></i><i class="bi bi-star-fill"></i><i class="bi bi-star-fill"></i><i class="bi bi-star-fill"></i><i class="bi bi-star-fill"></i>
            </div>
            <p>
              “This image compressor is a lifesaver for my blog. Uploads are lightning fast, and my site loads so much quicker. I love that there are no watermarks or limits!”
            </p>
            <div class="fw-bold" style="color:#2586f7;">– Emily, Travel Blogger</div>
          </div>
        </div>
        <div class="col-md-4">
          <div class="card shadow-sm p-4 h-100" style="background-color:#fff;color:#23272b;border:1px solid #e6e6e6;">
            <div class="mb-2 text-warning">
              <i class="bi bi-star-fill"></i><i class="bi bi-star-fill"></i><i class="bi bi-star-fill"></i><i class="bi bi-star-fill"></i><i class="bi bi-star-half"></i>
            </div>
            <p>
              “I use this tool every day for client projects. The batch compression feature saves me hours, and the quality is always spot on. Highly recommended for designers and developers.”
            </p>
            <div class="fw-bold" style="color:#2586f7;">– Raj, Web Designer</div>
          </div>
        </div>
        <div class="col-md-4">
          <div class="card shadow-sm p-4 h-100" style="background-color:#fff;color:#23272b;border:1px solid #e6e6e6;">
            <div class="mb-2 text-warning">
              <i class="bi bi-star-fill"></i><i class="bi bi-star-fill"></i><i class="bi bi-star-fill"></i><i class="bi bi-star-fill"></i><i class="bi bi-star-fill"></i>
            </div>
            <p>
              “As a photographer, I’m picky about image quality. This compressor keeps my photos sharp while making them small enough for fast sharing and uploads. It’s now a permanent part of my workflow.”
            </p>
            <div class="fw-bold" style="color:#2586f7;">– Lisa, Photographer</div>
          </div>
        </div>
      </div>
      <div class="row gx-5 gy-4 justify-content-center mt-4">
        <div class="col-md-4">
          <div class="card shadow-sm p-4 h-100" style="background-color:#fff;color:#23272b;border:1px solid #e6e6e6;">
            <div class="mb-2 text-warning">
              <i class="bi bi-star-fill"></i><i class="bi bi-star-fill"></i><i class="bi bi-star-fill"></i><i class="bi bi-star-fill"></i><i class="bi bi-star-fill"></i>
            </div>
            <p>
              “No more slow emails or failed uploads! I can compress images for my business presentations and newsletters in seconds. The interface is clean and super easy to use.”
            </p>
            <div class="fw-bold" style="color:#2586f7;">– Michael, Marketing Manager</div>
          </div>
        </div>
        <div class="col-md-4">
          <div class="card shadow-sm p-4 h-100" style="background-color:#fff;color:#23272b;border:1px solid #e6e6e6;">
            <div class="mb-2 text-warning">
              <i class="bi bi-star-fill"></i><i class="bi bi-star-fill"></i><i class="bi bi-star-fill"></i><i class="bi bi-star-fill"></i><i class="bi bi-star-fill"></i>
            </div>
            <p>
              “The drag-and-drop feature is a game changer. I can compress dozens of images for my online store in one go, and the results are always reliable. My site speed scores have never been better.”
            </p>
            <div class="fw-bold" style="color:#2586f7;">– Sofia, eCommerce Owner</div>
          </div>
        </div>
      </div>
    </div>
  </section>

  <!-- ORGANIC GROWTH TIPS -->
  <section class="wrapper bg-light">
    <div class="container pt-8 pt-md-10 pb-8 pb-md-12">
      <h2 class="display-4 mb-3">Tips for Effective Image Compression</h2>
      <div class="row gx-lg-8 gx-xl-12 gy-6">
        <div class="col-md-6 text-start">
          <ul class="icon-list bullet-bg bullet-soft-green fs-18 lh-lg">
            <li><i class="uil uil-check"></i><strong>Choose the Right Format:</strong> Use JPEG for photos, PNG for graphics, and WebP for the best balance of quality and size.</li>
            <li><i class="uil uil-check"></i><strong>Resize Before Compressing:</strong> Scale images to the exact size needed on your website or app to avoid unnecessary file weight.</li>
            <li><i class="uil uil-check"></i><strong>Preview Before Download:</strong> Always compare the original and compressed image to make sure quality meets your needs.</li>
            <li><i class="uil uil-check"></i><strong>Batch Compress for Efficiency:</strong> Save time by compressing multiple images at once, especially for large projects or eCommerce stores.</li>
            <li><i class="uil uil-check"></i><strong>Keep Originals Safe:</strong> Store uncompressed originals for future editing or printing, and use compressed versions for web and sharing.</li>
          </ul>
        </div>
        <div class="col-md-6 text-start">
          <ul class="icon-list bullet-bg bullet-soft-green fs-18 lh-lg">
            <li><i class="uil uil-check"></i><strong>Optimize Alt Text:</strong> Add descriptive alt tags to your images for better SEO and accessibility.</li>
            <li><i class="uil uil-check"></i><strong>Test Website Speed:</strong> Use tools like Google PageSpeed Insights to see how image compression improves your site’s load time.</li>
            <li><i class="uil uil-check"></i><strong>Automate Your Workflow:</strong> Integrate image compression into your CMS or build process for consistent optimization.</li>
            <li><i class="uil uil-check"></i><strong>Stay Updated:</strong> Follow best practices and new image formats to ensure your site stays fast and modern.</li>
          </ul>
        </div>
      </div>
      <div class="row mt-5">
        <div class="col-lg-10 mx-auto">
          <div class="alert alert-success p-4 fs-17">
            <strong>Did you know?</strong> Optimized images can reduce total page load time by up to 80%, leading to higher search rankings, more conversions, and happier users.
          </div>
        </div>
      </div>
    </div>
  </section>

  <!-- FAQ SECTION -->
  <section class="wrapper bg-gradient-primary">
    <div class="container pt-10 pt-md-12">
      <h2 class="display-4 mb-3">Frequently Asked Questions</h2>
      <div class="row gx-lg-8 gx-xl-12 gy-6">
        <div class="col-lg-6 text-start">
          <div class="accordion" id="faqAccordionLeft">
            <div class="accordion-item">
              <h2 class="accordion-header" id="faqHeading1">
                <button class="accordion-button collapsed" type="button" data-bs-toggle="collapse" data-bs-target="#faqCollapse1" aria-expanded="false" aria-controls="faqCollapse1">
                  How does the Image Compressor work?
                </button>
              </h2>
              <div id="faqCollapse1" class="accordion-collapse collapse" aria-labelledby="faqHeading1" data-bs-parent="#faqAccordionLeft">
                <div class="accordion-body">
                  <p>
                    The compressor uses advanced algorithms to reduce file size by removing unnecessary data and optimizing image encoding. It balances compression and quality, so your photos look sharp but load much faster.
                  </p>
                </div>
              </div>
            </div>
            <div class="accordion-item">
              <h2 class="accordion-header" id="faqHeading2">
                <button class="accordion-button collapsed" type="button" data-bs-toggle="collapse" data-bs-target="#faqCollapse2" aria-expanded="false" aria-controls="faqCollapse2">
                  Will my images lose quality?
                </button>
              </h2>
              <div id="faqCollapse2" class="accordion-collapse collapse" aria-labelledby="faqHeading2" data-bs-parent="#faqAccordionLeft">
                <div class="accordion-body">
                  <p>
                    Our tool is designed to minimize quality loss. In most cases, you’ll notice a huge reduction in file size with little to no visible change. You can preview the result before downloading to ensure it meets your standards.
                  </p>
                </div>
              </div>
            </div>
            <div class="accordion-item">
              <h2 class="accordion-header" id="faqHeading3">
                <button class="accordion-button collapsed" type="button" data-bs-toggle="collapse" data-bs-target="#faqCollapse3" aria-expanded="false" aria-controls="faqCollapse3">
                  What image formats are supported?
                </button>
              </h2>
              <div id="faqCollapse3" class="accordion-collapse collapse" aria-labelledby="faqHeading3" data-bs-parent="#faqAccordionLeft">
                <div class="accordion-body">
                  <p>
                    We support JPEG, PNG, WebP, GIF, and SVG formats. This covers almost all images used on websites, blogs, social media, and in print. More formats may be added in future updates.
                  </p>
                </div>
              </div>
            </div>
            <div class="accordion-item">
              <h2 class="accordion-header" id="faqHeading4">
                <button class="accordion-button collapsed" type="button" data-bs-toggle="collapse" data-bs-target="#faqCollapse4" aria-expanded="false" aria-controls="faqCollapse4">
                  Is there a file size or usage limit?
                </button>
              </h2>
              <div id="faqCollapse4" class="accordion-collapse collapse" aria-labelledby="faqHeading4" data-bs-parent="#faqAccordionLeft">
                <div class="accordion-body">
                  <p>
                    No, you can compress as many images as you like, of any size supported by your browser. There are no watermarks, no sign-up, and no hidden restrictions—just fast, free compression.
                  </p>
                </div>
              </div>
            </div>
            <div class="accordion-item">
              <h2 class="accordion-header" id="faqHeading5">
                <button class="accordion-button collapsed" type="button" data-bs-toggle="collapse" data-bs-target="#faqCollapse5" aria-expanded="false" aria-controls="faqCollapse5">
                  Is my privacy protected?
                </button>
              </h2>
              <div id="faqCollapse5" class="accordion-collapse collapse" aria-labelledby="faqHeading5" data-bs-parent="#faqAccordionLeft">
                <div class="accordion-body">
                  <p>
                    Yes, your images are processed securely and are never stored or shared. All compression happens in real-time, and your files are deleted immediately after processing.
                  </p>
                </div>
              </div>
            </div>
          </div>
        </div>
        <!--/col-->
        <div class="col-lg-6 text-start">
          <div class="accordion" id="faqAccordionRight">
            <div class="accordion-item">
              <h2 class="accordion-header" id="faqHeading6">
                <button class="accordion-button collapsed" type="button" data-bs-toggle="collapse" data-bs-target="#faqCollapse6" aria-expanded="false" aria-controls="faqCollapse6">
                  Can I compress multiple images at once?
                </button>
              </h2>
              <div id="faqCollapse6" class="accordion-collapse collapse" aria-labelledby="faqHeading6" data-bs-parent="#faqAccordionRight">
                <div class="accordion-body">
                  <p>
                    Absolutely! Use the batch compression feature to upload and optimize several images in one go. This is perfect for web projects, eCommerce stores, or anyone managing lots of visuals.
                  </p>
                </div>
              </div>
            </div>
            <div class="accordion-item">
              <h2 class="accordion-header" id="faqHeading7">
                <button class="accordion-button collapsed" type="button" data-bs-toggle="collapse" data-bs-target="#faqCollapse7" aria-expanded="false" aria-controls="faqCollapse7">
                  Will the tool work on my phone or tablet?
                </button>
              </h2>
              <div id="faqCollapse7" class="accordion-collapse collapse" aria-labelledby="faqHeading7" data-bs-parent="#faqAccordionRight">
                <div class="accordion-body">
                  <p>
                    Yes, our image compressor is fully responsive and works on all modern mobile devices and tablets. You can compress and download images directly from your phone, anywhere, anytime.
                  </p>
                </div>
              </div>
            </div>
            <div class="accordion-item">
              <h2 class="accordion-header" id="faqHeading8">
                <button class="accordion-button collapsed" type="button" data-bs-toggle="collapse" data-bs-target="#faqCollapse8" aria-expanded="false" aria-controls="faqCollapse8">
                  Does compression affect SEO?
                </button>
              </h2>
              <div id="faqCollapse8" class="accordion-collapse collapse" aria-labelledby="faqHeading8" data-bs-parent="#faqAccordionRight">
                <div class="accordion-body">
                  <p>
                    Yes—in a good way! Smaller images mean faster page loads, which Google rewards with higher rankings. Optimized images also improve user experience, which can boost your site’s SEO performance.
                  </p>
                </div>
              </div>
            </div>
            <div class="accordion-item">
              <h2 class="accordion-header" id="faqHeading9">
                <button class="accordion-button collapsed" type="button" data-bs-toggle="collapse" data-bs-target="#faqCollapse9" aria-expanded="false" aria-controls="faqCollapse9">
                  Can I use compressed images for printing?
                </button>
              </h2>
              <div id="faqCollapse9" class="accordion-collapse collapse" aria-labelledby="faqHeading9" data-bs-parent="#faqAccordionRight">
                <div class="accordion-body">
                  <p>
                    For web and digital use, compressed images are perfect. For high-quality printing, keep an original uncompressed version, as compression may reduce print quality at large sizes.
                  </p>
                </div>
              </div>
            </div>
            <div class="accordion-item">
              <h2 class="accordion-header" id="faqHeading10">
                <button class="accordion-button collapsed" type="button" data-bs-toggle="collapse" data-bs-target="#faqCollapse10" aria-expanded="false" aria-controls="faqCollapse10">
                  Is this tool really free?
                </button>
              </h2>
              <div id="faqCollapse10" class="accordion-collapse collapse" aria-labelledby="faqHeading10" data-bs-parent="#faqAccordionRight">
                <div class="accordion-body">
                  <p>
                    Yes! Our image compressor is 100% free to use, with no watermarks, usage limits, or hidden fees. Enjoy unlimited, high-quality image compression anytime you need it.
                  </p>
                </div>
              </div>
            </div>
          </div>
        </div>
        <!--/col-->
      </div>
      <!--/.row -->
      <div class="row mt-6">
        <div class="col-lg-10 mx-auto">
          <div class="alert alert-primary p-4 fs-17">
            <strong>Still have questions?</strong> Reach out to our support team anytime. We’re here to help you get the best results from your images!
          </div>
        </div>
      </div>
    </div>
  </section>
</div>
</body>
</html>

<?php include $_SERVER['DOCUMENT_ROOT'].'/admin/footer.php'; ?>
<script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.3/dist/js/bootstrap.bundle.min.js"></script>

<script>
function updateFilenameAndPreview(input) {
    var fileName = "No file chosen";
    var container = document.getElementById('orig-preview-container');
    if (container) container.innerHTML = '';
    if (input.files && input.files.length > 0) {
        fileName = input.files[0].name;
        // Show preview
        if (container) {
            var reader = new FileReader();
            reader.onload = function(e) {
                container.innerHTML = '<div class="img-preview-label">Preview</div><img src="'+e.target.result+'" class="img-fluid img-preview">';
            }
            reader.readAsDataURL(input.files[0]);
        }
    }
    document.getElementById('file-filename').textContent = fileName;
}
function enableFileInput() {
    var fileInput = document.getElementById('image_file');
    var fileNameDiv = document.getElementById('file-filename');
    var previewDiv = document.getElementById('orig-preview-container');
    fileInput.disabled = false;
    fileInput.value = '';
    fileNameDiv.textContent = 'No file chosen';
    if (previewDiv) previewDiv.innerHTML = '';
}
document.addEventListener('DOMContentLoaded', function() {
    var fileInput = document.getElementById('image_file');
    var fileNameDiv = document.getElementById('file-filename');
    var previewDiv = document.getElementById('orig-preview-container');
    if (fileInput && fileNameDiv) {
        if (!fileInput.disabled) {
            fileInput.value = '';
            fileNameDiv.textContent = 'No file chosen';
            if (previewDiv) previewDiv.innerHTML = '';
        }
        fileInput.onchange = function() {
            updateFilenameAndPreview(this);
        };
        fileNameDiv.onclick = function() {
            if (!fileInput.disabled) fileInput.click();
        };
    }
});
</script>